/*	
 Copyright 2013 Adobe Systems Incorporated.  All rights reserved. 

Purpose- 
This file has the implementation of Text editing functionality in Live Edit view
*/

/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global globalController, DW_LIBRARY_ITEM, ENTER_POST_PROCESSING_ACTIONS, CustomEvent, DW_LIVE_TEXT_EDIT, DW_LIVEEDIT_CONSTANTS, DW_LIVEEDIT_EVENTS, DW_EVENTS, DW_ICE_HEADLIGHTS, window, Node, TextEditHud, GenericHUDObject, DW_EDITABLE_REGION_CONSTS, document,dwExtensionController, liveViewExtensionsObject, DW_EXTENSION_EVENT */

//TextEditHud instantiation
TextEditHud.prototype = new GenericHUDObject();
TextEditHud.prototype.constructor = TextEditHud;
TextEditHud.prototype.baseClass = GenericHUDObject.prototype.constructor;

//TextEditHud class decleration
function TextEditHud(liveEditEvt) {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : constructor");
    this.setHudName(DW_LIVEEDIT_CONSTANTS.HudTextEdit);
    this.initialize(liveEditEvt);
    this.m_liveEditEvent.addListener(DW_LIVEEDIT_EVENTS.DoubleClick, this.enableHud, this);

    // TextEditHud class member initialization.
    this.m_elementUnderCurrentEditingSession = null;
    this.m_originalInnerHTMLOfElementUnderCurrentEditingSession = "";
    this.m_textEditFeedbackCss = DW_LIVEEDIT_CONSTANTS.OverlayCSS_TextEdit;

    // Current selected text node
    this.m_curTextNode = null;

	// This variable will have a reference to anchor tag if double click was on anchor tag or 
	// enter key is pressed when ESH is show on anchor
	this.m_nodeAtClickPoint = null;
	this.m_nodeAtClickPointInnerHTML = null;

    // Context which contains current text grouping info.
    this.m_textGroupContext = null;

    // Identification info to locate the text group in DW context.
    this.m_textIdentifier = null;

    // To track the any asyncronous request sent to DW
    this.m_dwReqCount = 0;

    // Updating text group would cause change in DW DOM tree.
    // Temp Id counter will be used to generate temp dw ids and 
    // to sync back the new data_liveedit_tagid values into live view DOM.
    this.m_dwgTempElementCount = 0;

    // Cache to reduce element search during editing process.
    this.m_nodeCache = null;

    // This signifies that there's a pending save document call to be done post commit and destroy
    this.saveDocumentPending = false;

    // To remember the encoded entities
    this.m_encodedEntities = {};
    // state of Formatting Hud
    this.m_lastSelection = null;
	// This flag specifies what post processing action to be performed on Enter key press.
	this.m_enterPostProcessingAction = null;

	// This flag holds the orginal tag name for tags who creates nested p tags on enter.
	this.m_blockLevelTagNameBeforeEnter = null;

	// static attributes - set by dwEditableCallback function.
	this.m_staticAttributes = null;

	// We introduce &nbsp; to make some elments editable
	// while entering into editing. Remember and remove them before
	// exiting from text editing session.
	this.m_mapTempNBSPsForEditing = {};

	// Some time we need to redraw the element when we exit from
	// text editing session.
	this.m_forceRedraw = false;
	this.m_elementInRedraw = null;
	this.m_displayStyleOfElementInRedraw = null;

	// Help Id for Text Hud
	this.HelpId = "AH_TEXT_HUD";
	// to track mosueout during selection
    this.isMouseDown = false;
}

TextEditHud.prototype.saveDocument = function () {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : saveDocument");
    this.saveDocumentPending = true;
};
/**
 * @override hasInlineStyle
 * check if the given style is in given element's inline style.
 * @param elem : obj element needs to be scanned
 * @param tag  : style stylename
 * @Return stylevalue or null if not present.
 */
TextEditHud.prototype.hasInlineStyle = function (obj, style) {
    'use strict';
	if (!obj || !style) {
        return null;
	}
    var attrs = obj.getAttribute('style'), x;
    if (attrs === null) {
        return null;
    }
    if (typeof attrs === 'object') {
        attrs = attrs.cssText;
    }
    var styles = attrs.split(';');
    for (x = 0; x < styles.length; x++) {
        var attr = styles[x].split(':')[0].replace(/^\s+|\s+$/g, "").toLowerCase();
        if (attr === style) {
            return styles[x].split(':')[1];
        }
    }
    return null;
};
/**
 * @override wrapChildrenWithTag
 * wrap the children of given element with the given tag name 
 * @param elem : Current Element 
 * @param tag  : tag with which we need to wrap children with.
 * @Return parentElement
 */
TextEditHud.prototype.wrapChildrenWithTag = function (elem, tag) {
    'use strict';
    if (!elem || !tag) {
        return;
    }
    var newNode = document.createElement(tag);
    if (!newNode) {
        return;
    }
    var childNodes = elem.childNodes;
    var i;
    if (childNodes) {
        for (i = 0; i < childNodes.length; i += 1) {
            var cloneNode = childNodes[i].cloneNode(true);
            newNode.appendChild(cloneNode);
            elem.removeChild(childNodes[i]);
        }
    }
    elem.appendChild(newNode);
    return elem;
};
/**
 * @override cleanUpContent
 * Depending on user preference. use b/strong, i/em tags
 * remove extra span tags attached by browser.
 * recursive call(MUST BE BOTTOM UP because we try to remove some nodes) that starts from dw_span and goes to all children of it.
 * @param curElement : Current Element 
 * @to-do need to respect the preference setting from dreamweaver
 */
TextEditHud.prototype.cleanUpContent = function (curElement) {
	'use strict';
    //do bottom up clearning for all nodes.
    if (!curElement) {
        return;
    }
    if (curElement.nodeType === Node.ELEMENT_NODE && curElement.tagName !== DW_LIVEEDIT_CONSTANTS.TextContainer) {
        var fontStyle = null, fontWeight = null;
        if (!curElement.getAttribute(DW_LIVEEDIT_CONSTANTS.FontStyleId) && !curElement.parentNode.getAttribute(DW_LIVEEDIT_CONSTANTS.FontStyleId)) {
            fontStyle = this.hasInlineStyle(curElement, "font-style");
            if (fontStyle) {
                fontStyle = fontStyle.trim();
                if (fontStyle === 'italic') {
                    curElement.style.removeProperty('font-style');
                    curElement = this.wrapChildrenWithTag(curElement, "i");
                } else if (fontStyle === 'normal') {
                    curElement.style.removeProperty('font-style');
                }
            }
        } else {
            fontStyle = this.hasInlineStyle(curElement, "font-style");
            if (!fontStyle) {
                var parentFont = this.hasInlineStyle(curElement.parentNode, "font-style");
                if (!parentFont) {
                    curElement.style["font-style"] = 'italic';
                }
            }
        }
        if (!curElement.getAttribute(DW_LIVEEDIT_CONSTANTS.FontWeightId) && !curElement.parentNode.getAttribute(DW_LIVEEDIT_CONSTANTS.FontWeightId)) {
            fontWeight = this.hasInlineStyle(curElement, "font-weight");
            if (fontWeight) {
                fontWeight = fontWeight.trim();
                if (fontWeight === 'bold') {
                    curElement.style.removeProperty('font-weight');
                    curElement = this.wrapChildrenWithTag(curElement, "b");
                } else if (fontWeight === 'normal') {
                    curElement.style.removeProperty('font-weight');
                }
            }
        } else {
            fontWeight = this.hasInlineStyle(curElement, "font-weight");
            if (!fontWeight) {
                var parentWeight = this.hasInlineStyle(curElement.parentNode, "font-weight");
                if (!parentWeight) {
                    curElement.style["font-weight"] = 'bold';
                }
            }
        }

        if (fontStyle || fontWeight) {
            var style = curElement.getAttribute("Style");
            if (style.trim() === "") {
				curElement.removeAttribute("Style");
			}
		}
    }
    var childNodes = curElement.childNodes;
    var i;
    if (childNodes) {
        for (i = 0; i < childNodes.length; i += 1) {
            this.cleanUpContent(childNodes[i]);
        }
    }

    if (curElement.nodeType === Node.ELEMENT_NODE) {
        curElement.removeAttribute(DW_LIVEEDIT_CONSTANTS.FontStyleId);
        curElement.removeAttribute(DW_LIVEEDIT_CONSTANTS.FontWeightId);
		if ((curElement.tagName === "B") && window.parent.semanticTags) {
			//replace with strong
			this.replaceTheTagName(curElement, "STRONG");
		} else if ((curElement.tagName === "I") && window.parent.semanticTags) {
			this.replaceTheTagName(curElement, "EM");
			//replace with em
		} else if (curElement.tagName === "SPAN") {
			var isExisting = curElement.getAttribute("DW_SPAN_ID");
            if (!isExisting) {
				//remove span tag
                this.removeNodeKeepChildren(curElement);
            } else {
				curElement.removeAttribute("DW_SPAN_ID");
			}
		}
	}
};
/**
 * @override replaceTheTagName
 * replace the given element's tag name with the newTagName
 * this is done mainly for the purpose of b/strong ,i/em tags.
 * browser inserts b tag but semantically strong tag is correct.
 * @param element : Current Element 
 * @param newTagName : Tag name with which we need to replace.
 */
TextEditHud.prototype.replaceTheTagName = function (element, newTagName) {
    'use strict';
	if (!element || !newTagName) {
        return;
    }
    var elemParent = element.parentNode;
    if (!elemParent) {
        return;
    }
    var semanticElement = window.parent.document.createElement(newTagName),
		attrs = element.attributes,
		i;
    if (attrs) {
        for (i = 0; i < attrs.length; i++) {
            semanticElement.setAttribute(attrs[i].nodeName, attrs[i].nodeValue);
        }
    }
    while (element.firstChild) {
        semanticElement.appendChild(element.firstChild);
    }
    elemParent.replaceChild(semanticElement, element);
};
/**
 * @private removeNodeKeepChildren
 * Remove the node  and attach it's children to it's parent. 
 * @param node : Node to be removed 
 */
TextEditHud.prototype.removeNodeKeepChildren = function (node) {
    'use strict';
    if (!node || !node.parentElement) {
        return;
    }
    while (node.firstChild) {
        node.parentElement.insertBefore(node.firstChild, node);
    }
    node.parentElement.removeChild(node);
};

/**
 * @private removeDynamicAttribute
 * Removes all dynamically populated attributes from element
 * @param elem : input element
 * attrMap: map that has static attributes of element
 */
TextEditHud.prototype.removeDynamicAttribute = function (elem, attrMap) {
    'use strict';

	this.logDebugMsg("removeDynamicAttribute from TextEditHud ");

	if (!elem || !attrMap) {
		return false;
	}

	if (elem.nodeType !== Node.ELEMENT_NODE) {
		return true;
	}

	var i,
		dwID = elem.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId),
		map = attrMap[dwID],
		removeList = [],
		setList = [];

	if (dwID && map) {
		// Find all attributes that needs to be removed or retained
		for (i = 0; i < elem.attributes.length; i++) {
			var a = elem.attributes[i];
			if (a.name !== DW_LIVEEDIT_CONSTANTS.DWTempId) {
				if (map[a.name]) {
					setList.push(a.name);
				} else {
					removeList.push(a.name);
				}
			}
		}

		// Remove all dynamically populated attributes
		for (i = 0; i < removeList.length; i++) {
			elem.removeAttribute(removeList[i]);
		}

		// Set all static attributes with original values
		for (i = 0; i < setList.length; i++) {
			elem.setAttribute(setList[i], map[setList[i]]);
		}
	}

	// Remove dynamic attributes in child nodes as well.
	var childNodes = elem.childNodes;
	for (i = 0; i < childNodes.length; i += 1) {
		this.removeDynamicAttribute(childNodes[i], attrMap);
	}

    return true;
};

/**
 * @private forceRedraw
 * Using hacky way to force redraw and relayouting for element.
 * @param elem : input element
 */
TextEditHud.prototype.forceRedraw = function (element) {
	'use strict';
    if (!element) {
		return;
	}

    var textNode = document.createTextNode(' '),
		disp = element.style.display;

    element.appendChild(textNode);
    element.style.display = 'none';

	this.m_elementInRedraw = element;
	this.m_displayStyleOfElementInRedraw = disp;

    setTimeout(function () {
        element.style.display = disp;
        textNode.parentNode.removeChild(textNode);

		// Just to make sure we don't leave empty style
		var style = element.getAttribute("Style");
		if (style.trim() === "") {
			element.removeAttribute("Style");
		}

		// When redraw happens while douple click on
		// some other element, it would have got affected
		// because of style.display = 'none'
		// Show overlay again.
		if (this.getVisibility()) {
			this.showTextEditingFeedback();
		}

		this.m_elementInRedraw = null;
		this.m_displayStyleOfElementInRedraw = null;
    }.bind(this), DW_LIVEEDIT_CONSTANTS.RedrawTimeOut);
};

// Override - commit - commits the changes back to user document.
TextEditHud.prototype.commit = function () {
    'use strict';

	this.logDebugMsg("function begin : TextEditHud : commit");

    if (this.getVisibility() === false) {
        return;
    }
    // Commit the changes back to user document.
    var textToCommitBack = null,
        dwSpanElement = this.m_elementUnderCurrentEditingSession,
		addedTempId = false;

    if (dwSpanElement) {
        if (window.liveViewExtensionsObject) {
            var isTextFormatHudVisible = false;
            isTextFormatHudVisible = window.liveViewExtensionsObject.isExtensionVisible("TextFormattingHud");
            if (isTextFormatHudVisible) {
                var txtFormatIframe = window.liveViewExtensionsObject.getExtensionIFrameById("TextFormattingHud");
                if (txtFormatIframe) {
                    var contWind = txtFormatIframe.contentWindow;
                    if (contWind && contWind.textFormattingHudObj) {
                        contWind.textFormattingHudObj.checkIfLinkChanged();
                        if (contWind.textFormattingHudObj.m_changedCode) {
                            // moved this function from text formatting hud to text edit. This removes extra span tag tags browser added,
                            // respects DW preference about strog/b tag.
                            // and also remove the span-id's that we added.

                            this.cleanUpContent(dwSpanElement);
                            this.clearUndoRedoOperations();
                            contWind.textFormattingHudObj.m_changedCode = false;
                        }
                    }
                }
            }
        }
        if (dwSpanElement.innerHTML !== this.m_originalInnerHTMLOfElementUnderCurrentEditingSession) {

            // If browser has left the any artefacts like &nbsps, duplicate Id attributes -- trim them off.
            this.cleanUpArtifactsOfEnterKeyPress(dwSpanElement);

            // When user selects all the text and deletes it,
            // browser inserts <br>. Ignoring it.
            this.removeUnwantedBRTag(dwSpanElement);

            // Create temp IDs for all nodes
            addedTempId = this.createTempIdS(dwSpanElement);

			// Clone node to do clean up before commiting.
			// We should not touch original element.
			var dwSpanClone  = dwSpanElement.cloneNode(true);

			// clean up &nbsp; that we added.
			this.cleanNBSPsIntroducedForEditing(dwSpanClone);

			// Remove dynamic attributes
			this.removeDynamicAttribute(dwSpanClone,  this.m_staticAttributes);

			// Text encoded with entity.
            textToCommitBack = this.getEntityEncodedInnerHTMLStr(dwSpanClone);

		} else if (this.saveDocumentPending) {
            window.DWSaveDocument();
            this.saveDocumentPending = false;
        }
    }

    if (textToCommitBack !== null) {
        if (!this.m_nodeCache) {
            this.m_nodeCache = {};
        }

        var idInfoObj = {},
            argObj = {},
            parentId = dwSpanElement.parentNode.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);

        this.m_nodeCache[parentId] = dwSpanElement.parentNode;

        // Temp ID information object
        idInfoObj.parentId = parentId;
        idInfoObj.callback = function (newIdObj) {
            this.updateDWIdForTempIDcallback(newIdObj);
        }.bind(this);

		if (addedTempId) {
			idInfoObj.lastTempId = this.m_dwgTempElementCount;
		}

        // Update temp Id Information object in identifier
        this.m_textIdentifier.tempIdInfoObj = idInfoObj;

        // Call DW Bridge function to update the text.
        argObj.textId = this.m_textIdentifier;
        argObj.textToCommit = textToCommitBack;
        window.DWLECallJsBridgingFunction(this.getHudName(), "DWSMUpdateLiveText", argObj, true);

		// Remember the inner HTML string to avoid recommit if any other commit call comes immediately.
		this.m_originalInnerHTMLOfElementUnderCurrentEditingSession = dwSpanElement.innerHTML;

		// Reposition all ER HUDs again and make them visible
        this.m_liveEditEvent.fire({
            type: DW_LIVEEDIT_EVENTS.ElementChanged
        });
    }

    GenericHUDObject.prototype.commit.call(this);
};


//  Override - draw - shows the editingFeedback UI around the element under edit 
//	and sets up the editing session
TextEditHud.prototype.draw = function (event, enteringTextEditingFromEnter) {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : draw ");

    if (this.getVisibility() === false) {
        return;
    }

    if (!this.m_curTextNode) {
        return;
    }

    // Update renderer that we are in editing.
    // This should be invoked before making any live editing related change in DOM
    // to allow the renderer to cache the Source/Selection and prevent source change 
    // notification while in edting.
    this.disableDWSelectionChanges(true);

    //clear current undo stack in browser
    this.clearUndoRedoOperations();

    // Create dw-span wrapper
    var dwSpanCreated = this.createDwSpanTextWrapper(this.m_curTextNode),
        element = this.getCurrentElement();

	// Show the editing feedback and setup editing session
    if (dwSpanCreated && element) {
		var ingoreIBeamPositioning = false;
		if (enteringTextEditingFromEnter) {
			ingoreIBeamPositioning = true;
		}

        this.enterTextEditingSession(element, ingoreIBeamPositioning);

        // Preserve the element which is under edit
        this.m_elementUnderCurrentEditingSession = element;

        // Preserve the element's original innerHTML, that will be used in case of cancelling the 
        // editing session via Esc key.
        this.m_originalInnerHTMLOfElementUnderCurrentEditingSession = element.innerHTML;
    } else {
        // Reset Text Identifying Context.
        this.resetContext();
        // Error in editing, reset the flag to allow Source/Selection sync.
        this.disableDWSelectionChanges(false);

		var elementUnderEdit = null;
		if (this.m_curTextNode.parentNode.tagName === DW_LIVEEDIT_CONSTANTS.TextContainer) {
			elementUnderEdit = this.m_curTextNode.parentNode.parentNode;
		} else {
			elementUnderEdit = this.m_curTextNode.parentNode;
		}
		if (element) {
			elementUnderEdit = this.FindElementToBringbackESHHud(element);
		}

		if (elementUnderEdit) {
			this.ShowEditingDisabledUI(elementUnderEdit);
		}

        // Bring back the ER.
        this.m_liveEditEvent.fire({
            type: DW_LIVEEDIT_EVENTS.EditOpFailed
        });
    }


    GenericHUDObject.prototype.draw.call(this);

    // Update NFW engine that text hud was shown
    var argObj = {};
    argObj.type = DW_LIVEEDIT_CONSTANTS.textEditedInLiveView;
    argObj.category = DW_LIVEEDIT_CONSTANTS.textEditEventCategory;
    window.DWLECallJsBridgingFunction(this.getHudName(), "sendNFWUpdate", argObj, false);

};


//  Override - destroy - removes the editing feedback UI
TextEditHud.prototype.destroy = function () {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : destroy");

    if (this.getVisibility() === false) {
        return;
    }
	var elementToBringbackESHHud = null;
    if (this.m_elementUnderCurrentEditingSession) {
		var parentNode = this.m_elementUnderCurrentEditingSession.parentNode;

		elementToBringbackESHHud = this.FindElementToBringbackESHHud(this.m_elementUnderCurrentEditingSession);
		this.exitTextEditingSession(this.m_elementUnderCurrentEditingSession);
        this.m_elementUnderCurrentEditingSession.outerHTML = this.m_elementUnderCurrentEditingSession.innerHTML;
        this.m_elementUnderCurrentEditingSession = null;

		// clean up &nbsp; that we added.
		this.cleanNBSPsIntroducedForEditing(parentNode);

		if (this.m_forceRedraw) {
			this.forceRedraw(parentNode);
		}
    }

    // Done with editing, reset the flag to allow Source/Selection sync.
    this.disableDWSelectionChanges(false);

    //clear undo operations caused by our editing
    this.clearUndoRedoOperations();

    // Reset Text Identifying Context after Escape handling.
    this.resetContext();
    // reset format hud state
    this.m_lastSelection = null;

	if (elementToBringbackESHHud) {
		var editDisabled = false;
		this.BringbackESHHud(elementToBringbackESHHud, editDisabled);
	}

	this.m_nodeAtClickPoint = null;
	this.m_nodeAtClickPointInnerHTML = null;

	GenericHUDObject.prototype.destroy.call(this);
};


// Override - escapeKeyPressed - removes the editing feedback UI and cancels the editing session
TextEditHud.prototype.escapeKeyPressed = function () {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : escapeKeyPressed");

    if (this.getVisibility() === false) {
        return;
    }

    var elementToBringbackESHHud = null;
    if (this.m_elementUnderCurrentEditingSession) {
		elementToBringbackESHHud = this.FindElementToBringbackESHHud(this.m_elementUnderCurrentEditingSession);
    }

    if (this.m_elementUnderCurrentEditingSession && this.m_originalInnerHTMLOfElementUnderCurrentEditingSession !== "") {
		var parentNode = this.m_elementUnderCurrentEditingSession.parentNode;

        this.exitTextEditingSession(this.m_elementUnderCurrentEditingSession);

        // Set back the original content - it removes the dw-span tag as well
        this.m_elementUnderCurrentEditingSession.outerHTML = this.m_originalInnerHTMLOfElementUnderCurrentEditingSession;

        this.m_elementUnderCurrentEditingSession = null;
        this.m_originalInnerHTMLOfElementUnderCurrentEditingSession = "";

		// clean up &nbsp; that we added.
		this.cleanNBSPsIntroducedForEditing(parentNode);

		if (this.m_forceRedraw) {
			this.forceRedraw(parentNode);
		}
    }

    // Done with editing, reset the flag to allow Source/Selection sync.
    this.disableDWSelectionChanges(false);

    //clear undo operations caused by our editing
    this.clearUndoRedoOperations();

    // Reset Text Identifying Context after Escape handling.
    this.resetContext();

    var messageDetails = {};
    messageDetails.type = DW_EXTENSION_EVENT.TEXT_EDIT_END;
    liveViewExtensionsObject.messageToExtensions(messageDetails);

    if (elementToBringbackESHHud) {
		var editDisabled = false;
		this.BringbackESHHud(elementToBringbackESHHud, editDisabled);
	}
    if (window.liveViewExtensionsObject && window.liveViewExtensionsObject.dwObject) {
        window.liveViewExtensionsObject.dwObject.logHeadlightsData(DW_ICE_HEADLIGHTS.DWShortcuts, DW_ICE_HEADLIGHTS.Canceled);
    }
    GenericHUDObject.prototype.destroy.call(this);
};

/*override
name: OnViewLostFocus
arguments: none

This function is called when browser view looses focus
*/

TextEditHud.prototype.OnViewLostFocus = function () {
    'use strict';

	this.logDebugMsg("function begin : TextEditHud : OnViewLostFocus");

    if (!GenericHUDObject.prototype.OnViewLostFocus.call(this)) {
        return;
    }

    this.CommitAndCloseHud();
};

TextEditHud.prototype.FindElementToBringbackESHHud = function (element) {
    'use strict';
	if (!element) {
		return;
	}

    this.logDebugMsg("function begin : TextEditHud : BringbackESHHud" + element);
	var elementForESH = null;
	if (element.firstChild && DW_LIVE_TEXT_EDIT.BlockInclusionList.indexOf(element.firstChild.tagName) >= 0) {
		elementForESH = element.firstChild;
	} else {
		elementForESH = element.parentNode;
	}

	return elementForESH;
};


TextEditHud.prototype.BringbackESHHud = function (element, editDisabled) {
    'use strict';
	if (!element) {
		return;
	}

    this.logDebugMsg("function begin : TextEditHud : BringbackESHHud" + element);

	var params = element.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);
	params += ",";
	params += "";
	params += ",";
	params += editDisabled;
	window.DWEventToLiveView(DW_EVENTS.SelectionChanged, params);
};

/**
 * @cleanUpArtifactsOfEnterKeyPress
 * Goes over all text nodes of the given element and trims 
 * white-spaces around each text elements.
 * @param node : Parent element node whose child textnodes are 
 * looked for trimming the text nodes.
 */
// Find first text node in the inside incoming node.
TextEditHud.prototype.cleanUpArtifactsOfEnterKeyPress = function (node) {
    'use strict';

    if (!node) {
        return;
    }
	this.logDebugMsg("TextEditHud.prototype.cleanUpArtifactsOfEnterKeyPress:" + node);

	this.trimWhiteSpacesArroundTextElements(node);
	this.removeDuplicateIdAttributes(node);
};


/**
 * @trimWhiteSpacesArroundTextElements
 * Goes over all text nodes of the given element and trims 
 * white-spaces around each text elements.
 * @param node : Parent element node whose child textnodes are 
 * looked for trimming the text nodes.
 */
// Find first text node in the inside incoming node.
TextEditHud.prototype.trimWhiteSpacesArroundTextElements = function (node) {
    'use strict';

    if (!node) {
        return;
    }

	this.logDebugMsg("TextEditHud.prototype.trimWhiteSpacesArroundTextElements:" + node);

    if (node.nodeType === Node.TEXT_NODE && node.nodeValue && node.nodeValue.length > 1) {
		if (node.nodeValue.charCodeAt(0) === DW_LIVEEDIT_CONSTANTS.NBSPCharCode) {
            node.nodeValue = " " + node.nodeValue.substr(1, node.nodeValue.length);
        }
        if (node.nodeValue.charCodeAt(node.nodeValue.length - 1) === DW_LIVEEDIT_CONSTANTS.NBSPCharCode) {
			node.nodeValue =  node.nodeValue.substr(0, node.nodeValue.length - 1) + " ";
        }

		return;
	}

    if (node.nodeType === Node.ELEMENT_NODE) {
        var chidNodes = node.childNodes;

        if (chidNodes) {
            var i, tempNode = chidNodes[0];
            for (i = 0; tempNode; i += 1, tempNode = chidNodes[i]) {
                this.trimWhiteSpacesArroundTextElements(tempNode);
            }
        }
	}
};


/**
 * @removeDuplicateIdAttributes
 * Goes over all elements nodes and removes all duplicate ids except the first node
 * @param node : Parent element node whose child element nodes to be looked for ID attribute
 * looked for trimming the text nodes.
 */
// Find first text node in the inside incoming node.
TextEditHud.prototype.removeDuplicateIdAttributes = function (node) {
    'use strict';

    if (!node) {
        return;
    }

	this.logDebugMsg("TextEditHud.prototype.removeDuplicateIdAttributes:" + node);

    var chidNodes = node.childNodes;
	if (chidNodes) {
		var i = 0,
			curNode = null,
			removeIdsFromNewlyAddedNodes = false;
		for (curNode = chidNodes[i]; curNode && removeIdsFromNewlyAddedNodes === false; i += 1, curNode = chidNodes[i]) {
			if (curNode.nodeType === Node.ELEMENT_NODE && curNode.hasAttribute(DW_LIVEEDIT_CONSTANTS.AttrbuteID)) {
				var curNodesIdAttribute = curNode.getAttribute(DW_LIVEEDIT_CONSTANTS.AttrbuteID);
				var nodeForward = null;
				for (nodeForward = curNode.nextSibling; nodeForward; nodeForward = nodeForward.nextSibling) {
					if (nodeForward.nodeType === Node.ELEMENT_NODE && nodeForward.hasAttribute(DW_LIVEEDIT_CONSTANTS.AttrbuteID) && (nodeForward.getAttribute(DW_LIVEEDIT_CONSTANTS.AttrbuteID) === curNodesIdAttribute)) {
						nodeForward.removeAttribute(DW_LIVEEDIT_CONSTANTS.AttrbuteID);
					}
				}

				if (nodeForward === null) {
					removeIdsFromNewlyAddedNodes = true;
				}
			}
		}
	}
};
// removeUnwantedBRTag - When you delete all text from editing session, <br> tag
// gets added automatically. This function is to clean that.
TextEditHud.prototype.removeUnwantedBRTag = function (dwSpanElement) {
	'use strict';
	if (!dwSpanElement) {
		return;
	}

	// When user selects all the text and deletes it, browser inserts <br>. 
	// If the text element which is under editing is only child for parent,
	// then we inserts &nbsp; to keep the tag editable in ELV.
	if (dwSpanElement.parentNode.firstChild === dwSpanElement.parentNode.lastChild) {
		if (dwSpanElement.innerHTML === "<br>" || dwSpanElement.innerHTML.length === 0) {
			dwSpanElement.innerHTML = DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
		}

		if ((dwSpanElement.parentNode.tagName === DW_LIVEEDIT_CONSTANTS.TableDataTagName ||
             dwSpanElement.parentNode.tagName === DW_LIVEEDIT_CONSTANTS.TableHeaderTagName) &&
				(dwSpanElement.innerHTML === "<br>" || dwSpanElement.innerHTML === DW_LIVEEDIT_CONSTANTS.NonBreakingSpace)) {
			dwSpanElement.innerHTML = DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
		}
	} else if (dwSpanElement.innerHTML === "<br>") {
		// #bug=3731578
		// We reach here when we select the complete content of hanging text, delete and commit the changes.
		// In this case of hanging text under edit, dwspan will not be only the child of its parent. 
		// The siblings could be new line elements or other valid tags itself.

		// we don't add nbsp; here, we delete the complete content.
		dwSpanElement.innerHTML = "";
	}

	// if single block child and its content is just '<br>', replace it
	// with "&nbsp;".
	var firstChild =  dwSpanElement.firstChild;
	if (firstChild && firstChild.nodeType === Node.ELEMENT_NODE &&
			firstChild === dwSpanElement.lastChild &&
			DW_LIVE_TEXT_EDIT.BlockInclusionList.indexOf(firstChild.tagName) >= 0 &&
			firstChild.innerHTML === "<br>") {
		firstChild.innerHTML = DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
	}

	// Still we have only <br>, remove it.
	if (dwSpanElement.innerHTML === "<br>") {
		dwSpanElement.innerHTML = "";
	}
};

// Checks for the special cases for which editing needs to be disabled.
TextEditHud.prototype.ShouldDisableEditing = function () {
	'use strict';

	var disablEditing = false;

	//Case 1 - If anchor no text and is styled with background image via CSS, disable editing.
	var element = this.m_nodeAtClickPoint;
	if ((element && element.tagName === "A") &&
			(this.m_nodeAtClickPointInnerHTML === "" || (this.m_nodeAtClickPointInnerHTML && this.m_nodeAtClickPointInnerHTML.trim() === ""))) {
		var elementsComputedStyle = window.getComputedStyle(element, null);
		if (elementsComputedStyle.background && elementsComputedStyle.background.indexOf("url") >= 0) {
			disablEditing = true;
		}
	}

	return disablEditing;
};

// Get the text node that is under the mouse click 
TextEditHud.prototype.getTextNodeFromClick = function (event) {
	'use strict';
	if (!event) {
		return null;
	}

	// Check if clicked element is text
    // As event.targetNode is always block level element, we need to
    // use display rect to findow out exact text element clicked.
    var range,
		x = event.clientX,
        y = event.clientY,
        elementOnClick = document.elementFromPoint(x, y);

    if (elementOnClick && elementOnClick.childNodes) {
        var i,
            foundTextNode = false,
            chidNodes = elementOnClick.childNodes,
			node = chidNodes[0];

		range = document.createRange();

        for (i = 0; node; i += 1, node = chidNodes[i]) {
            if (node.nodeType === Node.TEXT_NODE) {
                range.selectNode(node);
                var j,
                    rects = range.getClientRects(),
					rect = rects[0];

                for (j = 0; rect; j += 1, rect = rects[j]) {
                    if (x > rect.left && x < rect.right && y > rect.top && y < rect.bottom) {
                        elementOnClick = node;
                        foundTextNode = true;
                        break;
                    }
                }
            }
            if (foundTextNode) {
                break;
            }
        }
    }

	// If we found text node, set it
    if (elementOnClick.nodeType === Node.TEXT_NODE) {
        return elementOnClick;
    }

	return null;
};

// Is element inside the parent element
TextEditHud.prototype.isElementInsideParent = function (parent, searchnode) {
    'use strict';

    if (!parent || !searchnode) {
        return false;
    }

    if (parent === searchnode) {
        return true;
    }

    if (parent.nodeType === Node.ELEMENT_NODE) {
        var chidNodes = parent.childNodes;

        if (chidNodes) {
            var i, tempNode = chidNodes[0], found = false;
            for (i = 0; tempNode; i += 1, tempNode = chidNodes[i]) {
				if (tempNode === searchnode) {
					found = true;
					break;
				}

                found = this.isElementInsideParent(tempNode, searchnode);
                if (found) {
                    break;
                }
            }

			return found;
        }
    }

    return false;
};

// isEventInsideTextEditContext - Checks whether the click event is inside the Text Editing session.
TextEditHud.prototype.isEventInsideTextEditContext = function (event) {
    'use strict';

    this.logDebugMsg("TextEditHud.prototype.isEventInsideTextEditContext");

    if (!event || !this.getVisibility() || !this.m_elementUnderCurrentEditingSession) {
        return false;
    }

	var isInside = this.isElementInsideParent(this.m_elementUnderCurrentEditingSession, event.target);

	if (!isInside) {
		isInside = this.isElementInsideParent(this.m_elementUnderCurrentEditingSession, this.getTextNodeFromClick(event));
	}

	// If dw-span is only child for it's parent and event.target is of same, 
	// then we will consider this as inside context.
	if (!isInside && this.m_elementUnderCurrentEditingSession.parentNode === event.target &&
			this.m_elementUnderCurrentEditingSession.parentNode.firstChild === this.m_elementUnderCurrentEditingSession.parentNode.lastChild) {
		this.logDebugMsg("TextEditHud.isEventInsideTextEditContext Inside Context by Parent matching");
		isInside = true;
	}

	this.logDebugMsg("TextEditHud.isEventInsideTextEditContext is Inside? - " + isInside);

	return isInside;
};

// enterTextEditingSession - sets up the editing session for the input element 
TextEditHud.prototype.enterTextEditingSession = function (element, ingoreIBeamPositioning) {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : enterTextEditingSession");

    if (element) {
        window.DWLECallJsBridgingFunction(this.getHudName(), "clearLiveHighlight", null, false);
        window.DWLECallJsBridgingFunction(this.getHudName(), "setBoldItalicsPreference", null, false);
        this.addContentEditableAttribute(element);

        this.suspendEventPropogation(element);
        this.initializeTripleClickHandler(element);
        this.showTextEditingFeedback();
        this.initializeCutPasteHandler(element);
		// when something is selected inside text editing session and mouse up happened outside the text editing hud
		// still we should be able to propagate selection
        // this is mouseup listener to check if selection has changed and also removing the disable selection class in case it is added from mouseout
		window.addEventListener("mouseup", this.mouseupListenerOnWindow.bind(this), false);
        // Hide all ER HUDs
        this.m_liveEditEvent.fire({
            type: DW_LIVEEDIT_EVENTS.EditingText
        });
        // we need to tell formatting Hud about the node we are working on
        var messageDetails = {};
        messageDetails.type = DW_EXTENSION_EVENT.TEXT_EDIT_BEGIN;
        liveViewExtensionsObject.messageToExtensions(messageDetails);
        // reset format hud state
        this.m_lastSelection = null;
        var parent = element.parentNode;
        while (parent && parent.tagName !== "BODY") {
            if (parent.tagName === "TABLE") {
                if (window.liveViewExtensionsObject && window.liveViewExtensionsObject.dwObject) {
                    window.liveViewExtensionsObject.dwObject.logHeadlightsData(DW_ICE_HEADLIGHTS.OTH_ELV, DW_ICE_HEADLIGHTS.lveTableEdited);
                }
                break;
            }
            parent = parent.parentNode;
        }
        if (window.liveEditFGInstance) {
            if (window.liveViewExtensionsObject && window.liveViewExtensionsObject.dwObject) {
                window.liveViewExtensionsObject.dwObject.logHeadlightsData(DW_ICE_HEADLIGHTS.OTH_ELV, DW_ICE_HEADLIGHTS.lveFgEdited);
            }
        }


    } else {
        this.logDebugMsg("error : TextEditHud : enterTextEditingSession - failed");
    }
};


// exitTextEditingSession - exits the current editing session for the input element 
TextEditHud.prototype.exitTextEditingSession = function (element) {
    'use strict';

	this.logDebugMsg("function begin : TextEditHud : exitTextEditingSession");

    if (element) {
        this.clearTextEditingFeedback();
        this.removeContentEditableAttribute(element);
        this.resumeEventPropogation(element);
        this.unInitializeTripleClickHandler(element);
        this.uninitializeCutPasteHandler(element);
        window.removeEventListener("mouseup", this.mouseupListenerOnWindow.bind(this), false);
        element.removeEventListener("mouseout", this.mouseOutListener.bind(this), true);
        element.removeEventListener("mousedown", this.mouseDownListenerOnElement.bind(this), false);
        var messageDetails = {};
        messageDetails.type = DW_EXTENSION_EVENT.TEXT_EDIT_END;
        liveViewExtensionsObject.messageToExtensions(messageDetails);
    } else {
        this.logDebugMsg("error : TextEditHud : exitTextEditingSession - failed");
    }
};


// addContentEditableAttribute - adds contenteditable attribute to the input element &
//								disables the default UI of contenteditable attribute
TextEditHud.prototype.addContentEditableAttribute = function (element) {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : addContentEditableAttribute");
    if (element) {
        element.setAttribute("contenteditable", "true");
		// spl handling for PRE tag. In case we know more cases we can make it an array and traverse
        if (element.parentNode && element.parentNode.tagName === "PRE") {
            element.style.display = "inline-block";
        }
    } else {
        this.logDebugMsg("error : TextEditHud : addContentEditableAttribute - failed");
    }
};


// removeContentEditableAttribute - removes contenteditable attribute to the input element &
//								removes the default UI of contenteditable attribute
TextEditHud.prototype.removeContentEditableAttribute = function (element) {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : removeContentEditableAttribute");

    if (element) {
        element.removeAttribute("contenteditable");
        element.removeAttribute("style");
        element.blur();
    } else {
        this.logDebugMsg("error : TextEditHud : removeContentEditableAttribute - failed");
    }
};


// showTextEditingFeedback - shows the text editing feedback (orange border) around the 
// input element
TextEditHud.prototype.showTextEditingFeedback = function () {
    'use strict';

	// If the editing needs to be disabled, dont show it here.
	if (this.ShouldDisableEditing()) {
		return;
	}

    this.logDebugMsg("function begin : TextEditHud : showTextEditingFeedback");

    if (this.m_textEditFeedbackCss) {
        var paddingBetweenBorderAndContent = {
            left: DW_LIVEEDIT_CONSTANTS.OverlayPadding_Left,
            right: DW_LIVEEDIT_CONSTANTS.OverlayPadding_Right,
            top: DW_LIVEEDIT_CONSTANTS.OverlayPadding_Top,
            bottom: DW_LIVEEDIT_CONSTANTS.OverlayPadding_Bottom
        };

        this.showOverLayDiv(this.m_textEditFeedbackCss, paddingBetweenBorderAndContent);
    } else {
        this.logDebugMsg("error : TextEditHud : improper m_textEditFeedbackCss value");
    }
};


// clearTextEditingFeedback - removes the text editing feedback (orange border) around the 
// input element
TextEditHud.prototype.clearTextEditingFeedback = function () {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : clearTextEditingFeedback");

    this.hideOverLayDiv();
};

// positionIBeam - positions the IBeam to the event's x, y cordinates
TextEditHud.prototype.positionIBeam = function (event) {
	'use strict';

	this.logDebugMsg("function begin : TextEditHud : positionIBeam");

	if (!event) {
		return;
	}

	if (document.caretRangeFromPoint) {
		var x = event.clientX,
			y = event.clientY;

		if (x === null || y === null) {
			this.logDebugMsg("positionIBeam failed - invalid click position");
			return;
		}

		// IF element redraw is pending. make sure to set the display style as block before calucalating range and 
		// set the previous style after.
		var reDrawElementsStyle = null;
		if (this.m_elementInRedraw) {
			reDrawElementsStyle = this.m_elementInRedraw.style;
			this.m_elementInRedraw.style.display = "block";
		}

		var range = document.caretRangeFromPoint(x, y);

		// Element redraw is pending. set is previous style back.
		if (this.m_elementInRedraw && reDrawElementsStyle !== null) {
			this.m_elementInRedraw.style.display = reDrawElementsStyle;
		}

		if (range === null) {
			this.logDebugMsg("positionIBeam failed - invalid range received from caretRangeFromPoint");
			return;
		}

		if (range.startContainer.parentNode && this.hasSingleElementWhoseContentIsWhitespace(range.startContainer.parentNode)) {
			this.setIbeamToGivenCharOffsetInAGivenNode(range.startContainer, 0);
		} else {
			this.setIbeamToGivenCharOffsetInAGivenNode(range.startContainer, range.startOffset);
		}
	} else {
		this.logDebugMsg("positionIBeam failed - caretRangeFromPoint is not supported");
	}
};

// getEntityName - returns entity name for given character code.
//			if there is no name found in map, then it returns encoded '&#nnnn;' string
TextEditHud.prototype.getEntityName = function (code) {
    'use strict';

    if (!code) {
        return null;
    }

    var entName = this.m_encodedEntities[code];

    if (!entName) {
        entName = String.fromCharCode(code);
    }

    return entName;
};

//encodeString - Utility function that encodes all entities of given string and returns encoded string.
TextEditHud.prototype.encodeString = function (strToEncode) {
    'use strict';

    var encodedStr = strToEncode.replace(/[\u00A1-\u9999"']/gim, function (i) {
        return this.getEntityName(i.charCodeAt(0));
    }.bind(this));

    if (!encodedStr) {
        this.logDebugMsg('TextEditHud.encodeString - FAILED');
        return strToEncode;
    }

    return encodedStr;
};

//prepareEncodableEntities -  prepare encodable entities map.
TextEditHud.prototype.prepareEncodableEntities = function (originalText) {
    'use strict';

    // Reset map.
    this.m_encodedEntities = {};

    // Return if text is not valid.
    if (!originalText) {
        return;
    }

    // Find all valid entities and populate the map
    var idBegin = originalText.indexOf("&");
    while (idBegin >= 0) {
        var entityStr,
            entityCode = null,
            idEnd = originalText.indexOf(';', idBegin + 2);

        if (idEnd >= 0) {
            var hashChar = originalText.charAt(idBegin + 1);
            if (hashChar && hashChar === '#') {
                // Entity is encoded in "&#9999;" format.
                entityStr = originalText.substring(idBegin, idEnd + 1);
                entityCode = Number(originalText.substring(idBegin + 2, idEnd));
            } else {
                // Entity is encoded in "&quot;" format.
                entityStr = originalText.substring(idBegin, idEnd + 1);
                entityCode = DW_LIVE_TEXT_EDIT.Entities[entityStr];
            }
        }

        // Add valid entity into map and continue the search.
        if (entityCode) {
            this.m_encodedEntities[entityCode] = entityStr;
            idBegin = originalText.indexOf("&", idEnd + 1);
        } else {
            idBegin = originalText.indexOf("&", idBegin + 1);
        }
    }
};

//getEntityEncodedInnerHTMLStr - Encodes all entities of given node's inner html and returns.
TextEditHud.prototype.getEntityEncodedInnerHTMLStr = function (node) {
    'use strict';

    this.logDebugMsg('TextEditHud.getEntityEncodedInnerHTMLStr');
    if (!node) {
        return null;
    }

    var encodedStr = "";
    if (node.nodeType === Node.TEXT_NODE) {
        // plain text encode all the possible characters.
        encodedStr = this.encodeString(node.nodeValue);
    } else {
        //
        // Encode text element only:
        //		The inline HTML could be something like this
        //			<em>"some text <strong class="ddd">hello world</strong> dkijiuu"</em>
        //		Here, tag attributes (string inside "<" and ">") should not be encoded.
        //
        var orgStr = node.innerHTML,
            startIndex = 0,
            lastIndex = orgStr.length;

        while (startIndex < lastIndex) {
            var tagStartIndex = orgStr.indexOf("<", startIndex);
            if (tagStartIndex === -1) {
                var subStr = orgStr.substring(startIndex);
                if (subStr) {
                    this.logDebugMsg("Encode string (" + startIndex + ") :" + subStr);
                    encodedStr += this.encodeString(subStr);
                }
                break;
            }

            var tagEndIndex = orgStr.indexOf(">", tagStartIndex);
            if (tagEndIndex === -1) {
                // Some error case, we retain the string as it is.
                encodedStr = orgStr;
                break;
            }

            // Encode text string
            var subString = orgStr.substring(startIndex, tagStartIndex);
            if (subString) {
                this.logDebugMsg("Encode string (" + startIndex + " to " + tagStartIndex + ") :" + subString);
                encodedStr += this.encodeString(subString);
            }

            // Add tag string as it is.
            subString = orgStr.substring(tagStartIndex, tagEndIndex + 1);
            if (subString) {
                this.logDebugMsg("Append string :" + subString);
                encodedStr += subString;
            }

            startIndex = tagEndIndex + 1;
        }
    }

    return encodedStr;
};

//isInlineTextTag - returns true if the tagName is inline text tag neame
TextEditHud.prototype.isInlineTextTag = function (tagName) {
    'use strict';

    if (DW_LIVE_TEXT_EDIT.InlineTextElements.indexOf(tagName) >= 0) {
        return true;
    }
    return false;
};

//resetContext - clears the context information of current Text editing.
TextEditHud.prototype.resetContext = function () {
    'use strict';

    // Reset Text Identifying Context
    this.m_textGroupContext = null;
    this.m_textIdentifier = null;
	this.m_dwgTempElementCount = 0;
	this.m_staticAttributes = null;
	this.m_forceRedraw = false;

	this.m_nodeAtClickPoint = null;
	this.m_nodeAtClickPointInnerHTML = null;
};

//returns anchor node reference if the incoming node is child has an anchor node.
TextEditHud.prototype.findAnchorNodeParentForGivenNode = function (node) {
    'use strict';

	if (!node) {
		return null;
	}

    this.logDebugMsg("function begin : TextEditHud : findAnchorNodeParentForGivenNode:" + node);

	//Check if the incoming node is child has an anchor node.
	var anchorNode = null,
		nodeIter = node;
	while (nodeIter && nodeIter.tagName !== 'BODY') {
		if (nodeIter.tagName === 'A') {
			anchorNode = nodeIter;
			break;
		}
		nodeIter = nodeIter.parentNode;
	}

	//Check if the incoming node has direct and only child which is anchor.
	if (anchorNode === null) {
		if (node.firstChild && node.firstChild.tagName === 'A' && node.lastChild &&  node.firstChild === node.lastChild) {
			anchorNode = node.firstChild;
		}
	}

	return anchorNode;
};

//  Override - setCurrentElement - event Handler call back registerd with Controller. 
TextEditHud.prototype.setCurrentElement = function (args) {
    'use strict';

    // Validate arguments.
    this.logDebugMsg("function begin : TextEditHud : setCurrentElement");

    // Set / reset member's state.
    this.m_curElement = null;
    this.m_curTextNode = null;

    // clicking on the blank area or empty html should not disable PI, Insert Panel and Code View
    if (args && args.event) {
        var targetNode = args.event.target;
        if (targetNode && targetNode.nodeType === Node.ELEMENT_NODE && targetNode.tagName === "HTML") {
            return;
        }
    }

	// Element redraw is pending. make it visible before calculating
	// current element
	if (this.m_elementInRedraw) {
        this.m_elementInRedraw.style.display = this.m_displayStyleOfElementInRedraw;
	}

    if (!(args && args.element && args.event && args.isInEditableRegion)) {
        return;
    }

    var nodeAtClick = null;
	// Preserve the anchor element reference, if click in on the anchor tag.
	if (this.m_nodeAtClickPoint === null) {
		nodeAtClick = document.elementFromPoint(args.event.clientX, args.event.clientY);
		this.m_nodeAtClickPoint = this.findAnchorNodeParentForGivenNode(nodeAtClick);
		if (this.m_nodeAtClickPoint) {
			this.m_nodeAtClickPointInnerHTML = this.m_nodeAtClickPoint.innerHTML;
		}
	}

    this.m_curTextNode = this.getTextNodeFromClick(args.event);


    //if curr Text node is null, we will try to find it from event target node
    if (!this.m_curTextNode && args.event.target) {
		var element = args.event.target,
			curTextNode = this.findAnyTextNode(element);

		// If no text node, check whether there is any <br> tag
		if (!curTextNode) {
			curTextNode = this.findAnyBRNode(element);
		}

		// Check whether element can be used as text
		if (!curTextNode && this.canElementBeEditedAsText(element)) {
			curTextNode = element;
		}

		this.m_curTextNode = curTextNode;
    }

    if (this.m_curTextNode !== null) {
        if (window.liveViewExtensionsObject && window.liveViewExtensionsObject.dwObject) {
            window.liveViewExtensionsObject.dwObject.logHeadlightsData(DW_ICE_HEADLIGHTS.ELV_LVE, DW_ICE_HEADLIGHTS.StartEditing);
        }
        if (nodeAtClick && DW_LIVE_TEXT_EDIT.InlineTextElements.indexOf(nodeAtClick.tagName) >= 0) {
            if (window.liveViewExtensionsObject && window.liveViewExtensionsObject.dwObject) {
                window.liveViewExtensionsObject.dwObject.logHeadlightsData(DW_ICE_HEADLIGHTS.OTH_ELV, DW_ICE_HEADLIGHTS.InlineTagEdited_Pre + nodeAtClick.tagName + DW_ICE_HEADLIGHTS.InlineTagEdited_Post);
            }
        }
    }

	// Element redraw is pending. hide it again as we are done.
	if (this.m_elementInRedraw) {
        this.m_elementInRedraw.style.display = 'none';
	}
};

//
//getTextNodeIdentifier - returns text node identifier. Inline text like (strong, ...) will be 
//	identified by it's own "data_liveedit_tagid" attribute. Text node is identified by using following information.
//		parentID - parent node's data_liveedit_tagid
//		preNodeID - any immediate previous sibling element's data_liveedit_tagid
//		succNodeID - any immediate next sibling element's data_liveedit_tagid
//		noOfBeforeTexts - how many other text node exists in between preNodeID and current text Node.
//		noOfAfterTexts - how many other text node exists in between current text Node and succNodeID.
//		editableRegion - editable region name if it is template instance
//	
TextEditHud.prototype.getTextNodeIdentifier = function (nodeobj, ignoreDWSpanGrouping) {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : getTextNodeIdentifier");

    var textIdentifier = null;

    if (!nodeobj) {
        return textIdentifier;
    }

    if (nodeobj.nodeType === Node.ELEMENT_NODE && nodeobj.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId)) {
        textIdentifier = {};

        textIdentifier.nodeType = Node.ELEMENT_NODE;
        textIdentifier.dwId = nodeobj.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);

        return textIdentifier;
    }

    if (nodeobj.nodeType === Node.TEXT_NODE) {
        var parentNode, parentNodeID;

        // Get parent node info
        parentNode = nodeobj.parentNode;
        parentNodeID = parentNode.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);

		// dw_span tag is not real parent, in this case, take dw_span's parentNode.
		if (!parentNodeID && ignoreDWSpanGrouping && parentNode.tagName === DW_LIVEEDIT_CONSTANTS.TextContainer) {
			parentNodeID = parentNode.parentNode.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);
		}

        if (!parentNodeID) {
            return null;
        }

        //
        // Identify Sibling element information
        //
        var preNodeID = "",
            succNodeID = "",
            noOfBeforeTexts = 0,
            noOfAfterTexts = 0,
            editableRegion = null,
			parentPrevSiblingSearchDone = false,
			parentNextSiblingSearchDone = false,
            tempNode;

        // Get previous sibling element info
        tempNode = nodeobj.previousSibling;
        while (tempNode) {
            if (tempNode.nodeType === Node.TEXT_NODE && !this.isIgnorableText(tempNode)) {
                noOfBeforeTexts += 1;
            }
            if (tempNode.nodeType === Node.ELEMENT_NODE) {
                preNodeID = tempNode.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);
                break;
            }
            if (window.liveEditTemplateInstance && tempNode.nodeType === Node.COMMENT_NODE && tempNode.nodeValue.indexOf(DW_EDITABLE_REGION_CONSTS.EditableRegionBegin) === 0) {
                editableRegion = this.getEditableRegionAttributeVal(tempNode, DW_LIVEEDIT_CONSTANTS.DWUniqueId);
                break;
            }
            tempNode = tempNode.previousSibling;

			// When text node is directly under dw_span tag, all the previous sibling of dw_span tag needs to be considered.
			if (ignoreDWSpanGrouping && !tempNode && !parentPrevSiblingSearchDone && parentNode.tagName === DW_LIVEEDIT_CONSTANTS.TextContainer) {
				tempNode = parentNode.previousSibling;
				parentPrevSiblingSearchDone =  true;
			}
        }

        // Get next sibling element info
        tempNode = nodeobj.nextSibling;
        while (tempNode) {
            if (tempNode.nodeType === Node.TEXT_NODE && !this.isIgnorableText(tempNode)) {
                noOfAfterTexts += 1;
            }
            if (tempNode.nodeType === Node.ELEMENT_NODE) {
                succNodeID = tempNode.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);
                if (!succNodeID && tempNode.tagName.toLowerCase() === DW_LIVEEDIT_CONSTANTS.GlobalContainerDiv) {
                    succNodeID = "";
                }
                break;
            }
            if (window.liveEditTemplateInstance && tempNode.nodeType === Node.COMMENT_NODE && tempNode.nodeValue.indexOf(DW_EDITABLE_REGION_CONSTS.EditableRegionEnd) === 0) {
                break;
            }
            tempNode = tempNode.nextSibling;

			// When text node is directly under dw_span tag, all the next sibling of dw_span tag needs to be considered.
			if (ignoreDWSpanGrouping && !tempNode && !parentNextSiblingSearchDone && parentNode.tagName === DW_LIVEEDIT_CONSTANTS.TextContainer) {
				tempNode = parentNode.nextSibling;
				parentNextSiblingSearchDone =  true;
			}
        }

        // If any one of preNodeID or succNodeID is available, then we can identify the text relatively.
        if (preNodeID !== null || succNodeID !== null) {
            // Create textIdentifier object and populate information.
            textIdentifier = {};

            textIdentifier.nodeType = Node.TEXT_NODE;
            textIdentifier.parentNodeID = parentNodeID;
            textIdentifier.preNodeID = preNodeID;
            textIdentifier.succNodeID = succNodeID;
            textIdentifier.noOfBeforeTexts = noOfBeforeTexts;
            textIdentifier.noOfAfterTexts = noOfAfterTexts;
            if (editableRegion) {
                textIdentifier.editableRegion = editableRegion;
            }
        }
    }

    return textIdentifier;
};

//isTextGroup - utility function to check whether given node
//		is text node or text group ( block contains only inline text)
TextEditHud.prototype.isTextGroup = function (node) {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : isTextGroup");

    if (!node) {
        return false;
    }

    if (node.nodeType === Node.TEXT_NODE) {
        return true;
    }

    if (node.nodeType === Node.ELEMENT_NODE) {
        if (!node.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId)) {
            return false;
        }

        if (this.isInlineTextTag(node.tagName)) {
            return true;
        }
    }

    return false;
};

//isIgnorableText - checks whether given text node is dummy text
//		which can be ignored during sibling scanning
TextEditHud.prototype.isIgnorableText = function (node) {
    'use strict';

    if (!node || node.nodeType !== Node.TEXT_NODE) {
        return false;
    }

    // Is it ignorable Text?
    var i,
		ignorable = true,
        str = node.nodeValue,
        len = node.nodeValue.length;

    for (i = 0; i < len; i += 1) {
        if (DW_LIVE_TEXT_EDIT.WhiteSpaceChars.indexOf(str.charCodeAt(i)) < 0) {
            ignorable = false;
            break;
        }
    }

    return ignorable;
};

//groupNearbyTextSiblings - identifies near by sibling text nodes and returs 
//		group object which contains boundry information.
TextEditHud.prototype.groupNearbyTextSiblings = function (node) {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : groupNearbyTextSiblings");

    if (!node || !this.isTextGroup(node)) {
        return null;
    }

    //
    // Check previous siblings for grouping
    //
    var allPrevSiblingIsText = true,
        startingPrevSiblingText = node,
        previousSibling = node.previousSibling;
    while (previousSibling) {

        // break, if it is not text or text group
        if (!this.isTextGroup(previousSibling)) {
            allPrevSiblingIsText = false;
            break;
        }

        // Some time, if there is new line between to tag node, it is
        // comming as text node. Ignore that.
        if (this.isIgnorableText(previousSibling)) {
            // Ignore new line text node
            previousSibling = previousSibling.previousSibling;
        } else {
            // Remember the text sibling.
            startingPrevSiblingText = previousSibling;
            previousSibling = previousSibling.previousSibling;
        }
    }

    // check all next siblings
    var allNextSiblingIsText = true,
        endNextSiblingText = node,
        nextSibling = node.nextSibling;
    while (nextSibling) {

        // break, if it is not text or text group
        if (!this.isTextGroup(nextSibling)) {
            allNextSiblingIsText = false;
            break;
        }

        // Some time, if there is new line between to tag node, it is
        // comming as text node. Ignore that.
        if (this.isIgnorableText(nextSibling)) {
            // Ignore new line text node
            nextSibling = nextSibling.nextSibling;
        } else {
            // Remember the text sibling.
            endNextSiblingText = nextSibling;
            nextSibling = nextSibling.nextSibling;
        }
    }

    // create Group info object
    var textGroup = {};
    textGroup.originalNode = node;
    textGroup.startingSiblingText = startingPrevSiblingText;
    textGroup.endingSiblingText = endNextSiblingText;
    textGroup.allSiblingsAreText = allPrevSiblingIsText && allNextSiblingIsText;

    return textGroup;
};

// getBlockLevelGroup - checks whether block element can be included or not and
// returns group identification to include the block
TextEditHud.prototype.getBlockLevelGroup = function (nodeobj) {
	'use strict';
	this.logDebugMsg("function begin : TextEditHud : getBlockLevelGroup");

	// Need to consider only element node here.
	if (!nodeobj || nodeobj.nodeType !== Node.ELEMENT_NODE) {
		return null;
    }

	// Check whether we can include the block or not.
	if (nodeobj.innerHTML !== DW_LIVEEDIT_CONSTANTS.NonBreakingSpace &&
			DW_LIVE_TEXT_EDIT.BlockInclusionList.indexOf(nodeobj.tagName) < 0) {
		return null;
	}

	// create Group info object
    var textGroup = {};
    textGroup.originalNode = nodeobj;
    textGroup.startingSiblingText = nodeobj;
    textGroup.endingSiblingText = nodeobj;
    textGroup.allSiblingsAreText = false;
	textGroup.blockLevelGrouping = true;

	return textGroup;
};

// findTextGroupRange - find editable text group around given text node object
TextEditHud.prototype.findTextGroupRange = function (nodeobj) {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : findTextGroupRange");

    // Identify relative text Group
    var textGroup = this.groupNearbyTextSiblings(nodeobj);

    //
    // Group  at parent level if all siblings are text
    //
    if (textGroup && textGroup.allSiblingsAreText) {
        var parentTextGroup = this.findTextGroupRange(nodeobj.parentNode);
        if (parentTextGroup) {
            textGroup = parentTextGroup;
        } else {
			// if we are here, parentNode is block level element.
			// check whether we can include that in text group.
			parentTextGroup = this.getBlockLevelGroup(nodeobj.parentNode);
			if (parentTextGroup) {
				textGroup = parentTextGroup;
			}
		}
    }

    return textGroup;
};

//getTextIdentifierString - streams the textIdentifier properties as string and returns it. 
// Used for debugging purpose.
TextEditHud.prototype.getTextIdentifierString = function (textIdentifier) {
    'use strict';

    if (!textIdentifier) {
        return DW_LIVEEDIT_CONSTANTS.none;
    }

    var str = "";
    str += textIdentifier.nodeType;
    if (textIdentifier.nodeType === Node.TEXT_NODE) {
        str += " - ";
        str += textIdentifier.parentNodeID;
        str += " - ";
        str += textIdentifier.preNodeID;
        str += " - ";
        str += textIdentifier.succNodeID;
        str += " - ";
        str += textIdentifier.noOfBeforeTexts;
        str += " - ";
        str += textIdentifier.noOfAfterTexts;
        str += " - ";
        str += textIdentifier.editableRegion;
    } else {
        str += " - ";
        str += textIdentifier.dwId;
    }

    return str;
};

// prepareTextIdentifier - prepare text group context and Id for the given text node.
TextEditHud.prototype.prepareTextIdentifier = function (nodeobj, ignoreDWSpanGrouping) {
	'use strict';

    this.logDebugMsg("function begin : TextEditHud : prepareTextIdentifier");

    if (!nodeobj) {
        return false;
    }

    var nodeId = this.getTextNodeIdentifier(nodeobj, ignoreDWSpanGrouping);

    if (!nodeId) {
        return false;
    }

    //
    // Create and Initialize Text Group context
    //
    this.m_textGroupContext = {};
    this.m_textGroupContext.textNode = nodeobj;

	if (ignoreDWSpanGrouping) {
		this.m_textGroupContext.group = null;
	} else {
		this.m_textGroupContext.group = this.findTextGroupRange(nodeobj);
	}

    //
    // Prepare DW text identifier.
    //
    // Request Identity
    this.m_dwReqCount += 1;

    // Prepare async request to validate text selection with DW
    this.m_textIdentifier = nodeId;
    this.m_textIdentifier.id = this.m_dwReqCount;
    this.m_textIdentifier.callback = function (result, contextid, originalText, attrMap) {
        this.dwEditableCallback(result, contextid, originalText, attrMap);
    }.bind(this);
    this.m_textIdentifier.grouped = false;

	if (!this.m_textGroupContext.group && nodeobj.nodeType === Node.ELEMENT_NODE) {
		this.m_textGroupContext.group =  this.getBlockLevelGroup(nodeobj);
	}

	this.logDebugMsg("Text Node Id: " + this.getTextIdentifierString(this.m_textIdentifier));
    // Add group data to request;
    if (this.m_textGroupContext.group) {
        this.m_textIdentifier.grouped = true;
        this.m_textIdentifier.groupStarting = this.getTextNodeIdentifier(this.m_textGroupContext.group.startingSiblingText);
		this.logDebugMsg("Text Group Starting: " + this.getTextIdentifierString(this.m_textIdentifier.groupStarting));

        if (this.m_textGroupContext.group.startingSiblingText !== this.m_textGroupContext.group.endingSiblingText) {
            this.m_textIdentifier.groupEnding = this.getTextNodeIdentifier(this.m_textGroupContext.group.endingSiblingText);
			this.logDebugMsg("Text Group Ending: " + this.getTextIdentifierString(this.m_textIdentifier.groupEnding));
        }
    }

	return true;
};

// isInLibraryItem - checks whether the node is inside of Library Item.
TextEditHud.prototype.isInLibraryItem = function (nodeobj) {
	'use strict';

    this.logDebugMsg("function begin : TextEditHud : isInLibraryItem");

    if (!nodeobj) {
        return false;
    }

	var inLibrary = false,
		currNode = nodeobj;
	while (currNode) {
		if (currNode.nodeType === Node.COMMENT_NODE && currNode.nodeValue.indexOf(DW_LIBRARY_ITEM.Begin) === 0) {
			inLibrary = true;
			break;
		}

		if (currNode.nodeType === Node.COMMENT_NODE && currNode.nodeValue.indexOf(DW_LIBRARY_ITEM.End) === 0) {
			break;
		}

		if (currNode.previousSibling) {
			currNode = currNode.previousSibling;
		} else {
			currNode = currNode.parentNode;
		}
	}

	return inLibrary;
};

// createDwSpanTextWrapper - create dw-span tag wrapping text group to enable editing.
TextEditHud.prototype.createDwSpanTextWrapper = function (nodeobj) {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : createDwSpanTextWrapper");

    if (!nodeobj) {
        return false;
    }

    if (this.isInLibraryItem(nodeobj)) {
        if (window.liveViewExtensionsObject && window.liveViewExtensionsObject.dwObject) {
            window.liveViewExtensionsObject.dwObject.logHeadlightsData(DW_ICE_HEADLIGHTS.OTH_ELV, DW_ICE_HEADLIGHTS.LibEdited);
        }
        return false;
    }

	//
    // Prepare text group and identity for text node
    //
    if (!this.prepareTextIdentifier(nodeobj)) {
        return false;
    }

    //
    // Prepare text under dw-span tag
    //
    if (!this.prepareTextUnderDwSpan()) {
        return false;
    }

    //
    // Implemented asynchronous way of validating text selection.
    // The way HUD infrastructure validates the editablity is synchronous,
    // Text selection should be validated in aasynchronous way. The below callback is to 
    // cancel the editing on failure after validating with DW SM.
    //
    window.DWLECallJsBridgingFunction(this.getHudName(), "validateTextGroupForEdit", this.m_textIdentifier, false);

    return true;
};

// dwEditableCallback - call back function to be called from DW after validating
// editable range.
TextEditHud.prototype.dwEditableCallback = function (result, contextid, originalText, staticAttr) {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : dwEditableCallback - result: " + result);

	// If the editing is disabled explicitly, show the non-editable UI and bail.
	if (this.ShouldDisableEditing()) {
		this.handleValidationFailure();
	}

    if (contextid !== this.m_dwReqCount) {
        return;
    }

    if (result === true) {
        this.prepareEncodableEntities(originalText);
		this.m_staticAttributes = staticAttr;
        var strIdx, arrayLength = DW_LIVE_TEXT_EDIT.DynamicTypes.length;

        if (window.doctype) {
            for (strIdx = 0; strIdx < arrayLength; ++strIdx) {
                if (window.doctype.toUpperCase() === DW_LIVE_TEXT_EDIT.DynamicTypes[strIdx].toUpperCase()) {
                    if (window.liveViewExtensionsObject && window.liveViewExtensionsObject.dwObject) {
                        window.liveViewExtensionsObject.dwObject.logHeadlightsData(DW_ICE_HEADLIGHTS.OTH_ELV, DW_ICE_HEADLIGHTS.lvePreString + window.doctype + DW_ICE_HEADLIGHTS.lvePostStringStatic);
                    }
                    return true;
                }
            }
        }
    } else {
		this.handleValidationFailure();
    }
};

// Utility method for handling the validation. It clears the state and brings up ESH. 
TextEditHud.prototype.handleValidationFailure = function () {
    'use strict';

	// if text node range is not editable, alert user about dynamic changes
	// cancel the editing and show the feedback
	var elementUnderEdit = null;
	if (this.m_elementUnderCurrentEditingSession) {
		elementUnderEdit = this.FindElementToBringbackESHHud(this.m_elementUnderCurrentEditingSession);

		this.escapeKeyPressed();
	} else {
		var curElement = this.getCurrentElement();
		if (curElement) {
			curElement.outerHTML = curElement.innerHTML;
			this.m_curElement = null;
		}
	}

	//show non-editable UI and bail.
	this.ShowEditingDisabledUI(elementUnderEdit);

	this.m_liveEditEvent.fire({
		type: DW_LIVEEDIT_EVENTS.EditOpFailed
	});
};

// isFragmentHasLiteralSpaceAtBeginning - 
// returns : true if the incoming fragment starts with literal space characters
//		   : false if the fragment starts with any non-literal space (nbsp, br, newline or any other alphanumeric chars)
TextEditHud.prototype.isFragmentHasLiteralSpaceAtBeginning = function (fragment) {
    'use strict';

	if (!fragment) {
		return false;
	}

	this.logDebugMsg("function begin : TextEditHud : isFragmentHasLiteralSpaceAtBeginning" + fragment);

	var elementChildNodes = fragment.childNodes;
	if (elementChildNodes && elementChildNodes.length >= 1) {
		var nodeValueOfFirstChild = null;
		if (elementChildNodes[0].nodeType === Node.TEXT_NODE) {
			nodeValueOfFirstChild = elementChildNodes[0].nodeValue;
		} else if (elementChildNodes[0].nodeType === Node.ELEMENT_NODE) {
			nodeValueOfFirstChild = elementChildNodes[0].innerText;
		}

		if (nodeValueOfFirstChild && nodeValueOfFirstChild.charCodeAt(0) === DW_LIVEEDIT_CONSTANTS.SpaceKeyCode) {
			return true;
		}
	}

	return false;
};

// This function takes the fragment as input argument and checks its has single tag and the tag contains only white space
// as its content. White space may includes any number of new line, br tags and nbsps;
TextEditHud.prototype.hasSingleElementWhoseContentIsWhitespace = function (fragment) {
    'use strict';

	if (!fragment) {
		return false;
	}

	this.logDebugMsg("function begin : TextEditHud : hasSingleElementWhoseContentIsWhitespace" + fragment);

	var elementChildNodes = fragment.childNodes;
	if (elementChildNodes && elementChildNodes.length === 1) {
		return this.hasOnlyWhitespaceAsContent(elementChildNodes[0]);
	}

	return false;
};

// This function takes the element as input argument and checks that element contains only white space
// as its content. White space may includes any number of new line, br tags and nbsps;
TextEditHud.prototype.hasOnlyWhitespaceAsContent = function (element) {
    'use strict';

	if (!element) {
		return false;
	}

	this.logDebugMsg("function begin : TextEditHud : hasOnlyWhitespaceAsContent" + element);

	var nodeValueOfElement = null;
	if (element.nodeType === Node.TEXT_NODE) {
		nodeValueOfElement = element.nodeValue;
	} else if (element.nodeType === Node.ELEMENT_NODE) {
		nodeValueOfElement = element.innerText;
	}


	if (nodeValueOfElement) {
		nodeValueOfElement = nodeValueOfElement.trim();
		if (nodeValueOfElement.length === 0) {
			return true;
		}
	}

	return false;
};

TextEditHud.prototype.isFragmentHasOnlyOneBRTagAsItsContent = function (fragment) {
    'use strict';

	if (!fragment) {
		return false;
	}

	this.logDebugMsg("function begin : TextEditHud : isFragmentHasOnlyOneBRTagAsItsContent" + fragment);

	var elementChildNodes = fragment.childNodes,
		elementInnerHTML = null;
	if (elementChildNodes.length === 1) {
		var nodeValueOfFirstChild = null;
		if (elementChildNodes[0].nodeType === Node.TEXT_NODE) {
			nodeValueOfFirstChild = elementChildNodes[0].nodeValue;
		} else if (elementChildNodes[0].nodeType === Node.ELEMENT_NODE) {
			nodeValueOfFirstChild = elementChildNodes[0].innerText;
			elementInnerHTML = elementChildNodes[0].innerHTML;
		}

		if (nodeValueOfFirstChild) {
			// TODO - handle the case when fragment directly contains <br>
			if (nodeValueOfFirstChild.length === 1 && elementInnerHTML === "<br>") {
				return true;
			}
		}
	}

	return false;
};
// Identifies whether the ibeam is at the beginning of the incoming text node.
// returns true if the ibeam is at the beginning and false otherwise.
TextEditHud.prototype.isCaretAtBeginningOfTextNode = function (currentTextNode) {
	'use strict';

	if (!currentTextNode) {
		return false;
	}

	this.logDebugMsg("function begin : TextEditHud : isCaretAtBeginningOfTextNode :" + currentTextNode);

	var at_start = false,
		range = window.getSelection().getRangeAt(0),
		pre_range = document.createRange();

	if (!range) {
		return false;
	}

	pre_range.selectNodeContents(currentTextNode);
	pre_range.setEnd(range.startContainer, range.startOffset);

	var prior_text = pre_range.cloneContents();

	if (!prior_text) {
		return false;
	}

	// If the prior_text.textContent has literal space characters, they are not rendered in browser preview, hence we need to ignore
	// while calculating caret position.

	// If the prior_text.textContent has non-breaking space (&nbsp;), they are rendered in preview, hence we should not ignore them.
	var doTrim = this.isFragmentHasLiteralSpaceAtBeginning(prior_text);

	if (doTrim) {
		prior_text.textContent = prior_text.textContent.trim();
	}

	// If the text's length is 0, we're at the start of the text node.
	at_start = prior_text.textContent.length === 0;

	return at_start;
};


// Identifies whether the ibeam is at the end of the incoming text node.
// returns true if the ibeam is at the end and false otherwise.
TextEditHud.prototype.isCaretAtEndOfTextNode = function (currentTextNode) {
	'use strict';

	if (!currentTextNode) {
		return false;
	}

	this.logDebugMsg("function begin : TextEditHud : isCaretAtEndOfTextNode:" + currentTextNode);

	var at_end = false,
		range = window.getSelection().getRangeAt(0),
		post_range = document.createRange();

	if (!range) {
		return false;
	}

	post_range.selectNodeContents(currentTextNode);
	post_range.setStart(range.endContainer, range.endOffset);
	var next_text = post_range.cloneContents();
	next_text.textContent = next_text.textContent.trim();

	// If the text's length is 0, we're at the end of the text node.
	at_end = next_text.textContent.length === 0;
	return at_end;
};


// Utility method - the returns the first block level parent in the hierarchy for given text node.
TextEditHud.prototype.getFirstBlockLevelParent = function (node) {
	'use strict';

	if (!node || node.nodeType !== Node.ELEMENT_NODE) {
		return;
	}

	this.logDebugMsg("function begin : TextEditHud : getFirstBlockLevelParent:" + node);

	var nodeIter = node,
		blockLevelParent = null;
	while (nodeIter) {
		if (DW_LIVE_TEXT_EDIT.BlockInclusionList.indexOf(nodeIter.tagName) >= 0 || nodeIter.tagName === DW_LIVEEDIT_CONSTANTS.TextContainer) {
			break;
		} else {
			nodeIter = nodeIter.parentNode;
		}
	}

	if (nodeIter) {
		blockLevelParent = nodeIter;
	}

	return blockLevelParent;
};

// has the given Tag as child?
TextEditHud.prototype.elementContainTheTag = function (node, tagName) {
	'use strict';
	if (node && node.nodeType === Node.ELEMENT_NODE) {
		if (node.tagName === tagName) {
			return true;
		}

        var chidNodes = node.childNodes;
        if (chidNodes) {
            var i, tempNode = chidNodes[0];
            for (i = 0; tempNode; i += 1, tempNode = chidNodes[i]) {
				if (this.elementContainTheTag(tempNode, tagName)) {
					return true;
				}
            }
        }
    }
	return false;
};

// Identifies whether to do any post processing after browser handles enter key.
// This is called from onKeyDown - i.e., before browser handles enter key. By looking
// the ibeam position, we identify different actions to perform in on keyup.
TextEditHud.prototype.FigureOutPostProcessingAfterEnterKeyPress = function (evt) {
	'use strict';

	this.logDebugMsg("function begin : TextEditHud : FigureOutPostProcessingAfterEnterKeyPress");
	try {
		var node = document.getSelection().anchorNode;
		if (!node) {
			return;
		}

		var nodeActual =  (node.nodeType === Node.TEXT_NODE ? node.parentNode : node);
		if (!nodeActual) {
			return;
		}

        if (nodeActual.tagName === "MAIN" && nodeActual.parentElement && nodeActual.parentElement.tagName === DW_LIVEEDIT_CONSTANTS.TextContainer) {
            if (!evt) {
                evt = window.event;
            }
            evt.stopPropagation();
            evt.preventDefault();
            this.m_enterPostProcessingAction = null;
			return;
        }
		var inTagThatCreatesNestedPTagsOnEnter = false;
		if (DW_LIVE_TEXT_EDIT.TagsThatCreatesPTagsOnEnter.indexOf(nodeActual.tagName) >= 0) {
			inTagThatCreatesNestedPTagsOnEnter = true;
			this.m_blockLevelTagNameBeforeEnter = nodeActual.tagName;
		}

		nodeActual = this.getFirstBlockLevelParent(nodeActual);

		if (DW_LIVE_TEXT_EDIT.LeaveEnterKeyPressBehaviourToBrowser.indexOf(nodeActual.tagName) >= 0) {
			this.m_enterPostProcessingAction = null;
			return;
		}

		var lookForEnterProcessing = false;
		if (nodeActual.tagName === DW_LIVEEDIT_CONSTANTS.TextContainer && DW_LIVE_TEXT_EDIT.TagsThatCreatesPTagsOnEnter.indexOf(nodeActual.parentNode.tagName) >= 0) {
			document.execCommand('formatBlock', false, 'p');
			lookForEnterProcessing = true;
		}

		var inListItemTag = false;
		if (nodeActual.tagName === "LI") {
			inListItemTag = true;
		}

		if (DW_LIVE_TEXT_EDIT.BlockInclusionList.indexOf(nodeActual.tagName) >= 0) {
			lookForEnterProcessing = true;
		}

		var tagIsInline = false;
		if (DW_LIVE_TEXT_EDIT.InlineTextElements.indexOf(nodeActual.tagName) >= 0) {
			tagIsInline = true;
			lookForEnterProcessing = true;
		}

		if (lookForEnterProcessing === true) {
			if (inTagThatCreatesNestedPTagsOnEnter === true) {
				this.m_enterPostProcessingAction = ENTER_POST_PROCESSING_ACTIONS.CreateNestedPTagsAndInsertThemToOrignalParent;
			} else if (this.isCaretAtBeginningOfTextNode(nodeActual)) {
				this.m_enterPostProcessingAction = ENTER_POST_PROCESSING_ACTIONS.ChangePreviousSiblingContentToNBSP;
			} else if (this.isCaretAtEndOfTextNode(nodeActual)) {
				if (inListItemTag) {
					this.m_enterPostProcessingAction = ENTER_POST_PROCESSING_ACTIONS.ChangeCurrentNodeContentToNBSP;
				} else {
					var needsPostProcessing = true;
					// if ibeam is currently at the end of in line tag, do post processing ONLY if it happens to be last child of its parent.
					if (tagIsInline === true) {
						if (nodeActual.parentNode && nodeActual.parentNode.lastChild &&
								nodeActual !== nodeActual.parentNode.lastChild) {
							needsPostProcessing = false;
						}
					}

					if (needsPostProcessing === true) {
						this.m_enterPostProcessingAction = ENTER_POST_PROCESSING_ACTIONS.ChangeCurrentNodeToParaAndChangeContentToNBSP;
					}
				}
			}
		}
	} catch (e) {
        this.logDebugMsg("FigureOutPostProcessingAfterEnterKeyPress - exception caught:" + e.message);
    }

};

// Performs post processing action identified by m_enterPostProcessingAction.
// This is called from onKeyUp i.e., after browsers enter key handling.
TextEditHud.prototype.DoPostProcessingAfterEnterKeyPress = function () {
	'use strict';

    if (this.m_enterPostProcessingAction === null) {
		return;
	}
    this.logDebugMsg("function begin : TextEditHud : DoPostProcessingAfterEnterKeyPress" + this.m_enterPostProcessingAction);

	var node = document.getSelection().anchorNode;
	if (!node) {
		return;
	}

	var nodeActual =  (node.nodeType === Node.TEXT_NODE ? node.parentNode : node);
	if (!nodeActual) {
		return;
	}

	var innerHTML;
	if (this.m_enterPostProcessingAction === ENTER_POST_PROCESSING_ACTIONS.ChangeCurrentNodeContentToNBSP) {
		var currentNode = nodeActual;
		if (currentNode && !this.elementContainTheTag(currentNode, "IMG")) {
			currentNode.innerHTML = DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;

			this.setIbeamToGivenCharOffsetInAGivenNode(currentNode, 0);
		}
	} else if (this.m_enterPostProcessingAction === ENTER_POST_PROCESSING_ACTIONS.ChangePreviousSiblingContentToNBSP) {
		var previousSiblingNode = null,
			nodeActualsFirstBlockLevelParent = this.getFirstBlockLevelParent(nodeActual);
		if (nodeActualsFirstBlockLevelParent) {
			previousSiblingNode = nodeActualsFirstBlockLevelParent.previousSibling;
		}

		if (previousSiblingNode && !this.elementContainTheTag(previousSiblingNode, "IMG")) {
			previousSiblingNode.innerHTML = DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
		}

		if (previousSiblingNode && previousSiblingNode.nextSibling && this.elementContainTheTag(previousSiblingNode, "IMG")) {
			innerHTML = previousSiblingNode.nextSibling.innerHTML.trim();
			if (innerHTML === "" || innerHTML === "<br>") {
				previousSiblingNode.nextSibling.innerHTML = DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
			}
		}
	} else if (this.m_enterPostProcessingAction === ENTER_POST_PROCESSING_ACTIONS.ChangeCurrentNodeToParaAndChangeContentToNBSP) {
		var targetElementToReplace = nodeActual;
		if (DW_LIVE_TEXT_EDIT.InlineTextElements.indexOf(nodeActual.tagName) >= 0) {
			targetElementToReplace = this.getFirstBlockLevelParent(nodeActual);

			// if targetElementToReplace is dw-span, then our actual node to replace is the div which got added browser.
			// start from nodeActual and go upwards till dw-span and break when you find a div which doesnot have data_liveedit_tagid
			if (targetElementToReplace && targetElementToReplace.tagName === DW_LIVEEDIT_CONSTANTS.TextContainer) {
				var nodeIter = nodeActual;
				while (nodeActual !== targetElementToReplace) {
					if (nodeIter.tagName === "DIV" && !nodeIter.hasAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId)) {
						targetElementToReplace = nodeIter;
						break;
					}
					nodeIter = nodeIter.parentNode;
				}
			}
		}

		if (targetElementToReplace) {
			if (!this.elementContainTheTag(targetElementToReplace, "IMG")) {
				targetElementToReplace.innerHTML = DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
			}

			// Unlike h1-h5, if enter is pressed at the end of H6 tag, browser doesn't create div, instead it creates H5. 
			if (targetElementToReplace.tagName === "DIV" || targetElementToReplace.tagName === "H6") {
				document.execCommand('formatBlock', false, 'p');
			} else {
				this.setIbeamToGivenCharOffsetInAGivenNode(targetElementToReplace, 0);
			}

			if (targetElementToReplace && targetElementToReplace.previousSibling && this.elementContainTheTag(targetElementToReplace, "IMG")) {
				innerHTML = targetElementToReplace.previousSibling.innerHTML.trim();
				if (innerHTML === "" || innerHTML === "<br>") {
					targetElementToReplace.previousSibling.innerHTML = DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
				}
			}
		}
	} else if (this.m_enterPostProcessingAction === ENTER_POST_PROCESSING_ACTIONS.CreateNestedPTagsAndInsertThemToOrignalParent) {
		if (this.m_blockLevelTagNameBeforeEnter) {
			var firstBlockLevelNode = this.getFirstBlockLevelParent(nodeActual);
			if (!firstBlockLevelNode || !firstBlockLevelNode.previousSibling) {
				return;
			}

			var origNode = document.createElement(this.m_blockLevelTagNameBeforeEnter);
			var range = document.createRange();
			range.setStartBefore(firstBlockLevelNode.previousSibling);
			range.setEndAfter(firstBlockLevelNode);
			range.surroundContents(origNode);

			if (firstBlockLevelNode.previousSibling.innerHTML === "<br>") {
				firstBlockLevelNode.previousSibling.innerHTML = DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
			}

			if (firstBlockLevelNode.innerHTML === "<br>") {
				firstBlockLevelNode.innerHTML = DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
			}

			this.setIbeamToGivenCharOffsetInAGivenNode(firstBlockLevelNode, 0);
			this.m_blockLevelTagNameBeforeEnter = null;
		}
	}

	var selectionChangeEvent = new CustomEvent("selectionchange");
    document.dispatchEvent(selectionChangeEvent);
    if (window.liveViewExtensionsObject && window.liveViewExtensionsObject.dwObject) {
        window.liveViewExtensionsObject.dwObject.logHeadlightsData(DW_ICE_HEADLIGHTS.ELV_LVE, DW_ICE_HEADLIGHTS.NewTagCreatedViaEnter);
    }
};

// Utility method - sets IBeam to the start of the incoming node.
TextEditHud.prototype.setIbeamToGivenCharOffsetInAGivenNode = function (node, charOffset) {
	'use strict';

	if (!node) {
		return;
	}

	this.logDebugMsg("function begin : TextEditHud : setIbeamToGivenCharOffsetInAGivenNode:" + node);

	var iBeamPos = document.createRange();
	iBeamPos.setStart(node, charOffset);
	iBeamPos.setEnd(node, charOffset);
	iBeamPos.collapse(false);

	var selectionObject = window.getSelection();
	selectionObject.removeAllRanges();
	selectionObject.addRange(iBeamPos);

	// tell text formatting hud about this selection Change
	this.selectHandler();
};


// A utility method that inserts incoming 'html' at the cursor point.
// Currently this is called for adding <br> tag on Shift+Enter key press.
TextEditHud.prototype.insertHtmlAtCaretPosition = function (html) {
	'use strict';

	if (!html) {
		return;
	}

	this.logDebugMsg("function begin : TextEditHud : insertHtmlAtCaretPosition:" + html);

    var sel = window.getSelection(),
		range;

	if (!sel) {
		return;
	}

	if (sel.getRangeAt && sel.rangeCount) {
		range = sel.getRangeAt(0);
		range.deleteContents();

		//Create a temp div to contain the incoming html
		var el = document.createElement("div");
		el.innerHTML = html;

		//Create a temp fragment and insert at the range
		//TODO - currently this function is used for inserting <br> tag, that's why we are interested 
		// only in first child. Improve this if this method needs to work for multiple nodes html.
		var frag = document.createDocumentFragment(),
			node = el.firstChild,
			lastNode;

		if (node) {
			lastNode = frag.appendChild(node);
		}

		// Insert the fragment which contains incoming html at the range.
		range.insertNode(frag);

		// Set the selection to the end of newly added html
		if (lastNode) {
			range = range.cloneRange();
			range.setStartAfter(lastNode);
			range.collapse(true);
			sel.removeAllRanges();
			sel.addRange(range);
		}
	}
};

// prepareTextUnderDwSpan - move text group under dw-span and insert it
//		live view DOM replacing the original text group.
TextEditHud.prototype.prepareTextUnderDwSpan = function () {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : prepareTextUnderDwSpan");
    if (!this.m_textGroupContext) {
        return false;
    }

    var dwSpanNode = document.createElement(DW_LIVEEDIT_CONSTANTS.TextContainer);

    // Text Group or individual text node? 
    if (this.m_textGroupContext.group && this.m_textGroupContext.group.startingSiblingText && this.m_textGroupContext.group.endingSiblingText) {

        var currNode = this.m_textGroupContext.group.startingSiblingText,
            parentNode = currNode.parentNode,
            endNode = this.m_textGroupContext.group.endingSiblingText;

        // insert dw-span into DOM tree.
        parentNode.insertBefore(dwSpanNode, currNode);

        // move all text nodes as child into dw-span
        var clonedNode;
        while (currNode) {
            var nextNode = currNode.nextSibling,
                foundEndNode = false;

            if (currNode === endNode) {
                foundEndNode = true;
            }

            clonedNode = parentNode.removeChild(currNode);
            dwSpanNode.appendChild(clonedNode);

            currNode = nextNode;

            if (foundEndNode) {
                break;
			}
        }
    } else {
		// Individual text node just wrap it with dw-span.
		// Sometime, text node value is null when tag is empty.
		var nodeobj = this.m_textGroupContext.textNode;
		if (!nodeobj || !nodeobj.nodeValue) {
			return false;
		}

		dwSpanNode.appendChild(document.createTextNode(nodeobj.nodeValue));
        nodeobj.parentNode.replaceChild(dwSpanNode, nodeobj);
    }

    // Update the current element to dw-span node.
    this.m_curElement = dwSpanNode;

	// We need to redraw parent element some times when we 
	// exit from text editing session
	this.m_forceRedraw = this.m_curElement &&
						this.m_curElement.firstChild &&
						this.m_curElement.firstChild.nodeType === Node.ELEMENT_NODE &&
						DW_LIVE_TEXT_EDIT.TagsNeedsRedraw.indexOf(this.m_curElement.firstChild.tagName) >= 0;

	// To re-sizing the feedback overlay when we type characters.
    dwSpanNode.onkeyup = function (event) {
		if (event.keyCode === DW_LIVEEDIT_CONSTANTS.EnterKeyCode  && event.shiftKey) {
			this.insertHtmlAtCaretPosition('<br>');
			event.preventDefault();
			event.stopPropagation();
		} else if (event.keyCode === DW_LIVEEDIT_CONSTANTS.EnterKeyCode && this.m_enterPostProcessingAction !== null) {
			this.DoPostProcessingAfterEnterKeyPress();
			this.m_enterPostProcessingAction = null;
		} else if (event.keyCode === DW_LIVEEDIT_CONSTANTS.BackSpace) {
            if (this.m_curElement) {
                var firstCh = this.m_curElement.firstChild;
                if (firstCh && firstCh.nodeType === Node.ELEMENT_NODE && firstCh.tagName === DW_LIVEEDIT_CONSTANTS.TableDataTagName) {
                    var innerHtm = firstCh.innerHTML.trim();
                    if (innerHtm.length === 0 || innerHtm === "<br>") {
                        firstCh.innerHTML = DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
                    }
                    return;
                }
            }
            if (this.hasSingleElementWhoseContentIsWhitespace(this.m_curElement) && this.isCaretAtBeginningOfTextNode(this.m_curElement)) {
				var collaspeTag = true;
				// we should not delete the tag if the element has only one <br> tag. When user deletes all the tag content, browser adds <br> tag when last character is deleted. Hence we should not delete the tag when last char is deleted.
				if (this.isFragmentHasOnlyOneBRTagAsItsContent(this.m_curElement)) {
					collaspeTag = false;
				}
				if (collaspeTag) {
					this.m_curElement.innerHTML = "";
				}
			}

			if (this.m_curElement.innerHTML.length === 0) {
				window.CommitAndCloseHud();
				return;
			}
		} else if (event.keyCode === DW_LIVEEDIT_CONSTANTS.DeleteKey) {
			if (this.m_curElement.innerHTML.length === 0) {
				this.m_curElement.innerHTML = "<br>";
			}
		}
		this.showTextEditingFeedback();
        this.selectHandler();
    }.bind(this);
	// mousedown is to tracked selection on mouseout
    dwSpanNode.addEventListener("mousedown", this.mouseDownListenerOnElement.bind(this), false);
    // mouseout should disable selection outside and let the selection be there for our element
	dwSpanNode.addEventListener("mouseout", this.mouseOutListener.bind(this), true);
    dwSpanNode.onkeydown = function (event) {
		var shiftDown = event.shiftKey;
		if (event.keyCode === DW_LIVEEDIT_CONSTANTS.EnterKeyCode && !shiftDown) {
			this.FigureOutPostProcessingAfterEnterKeyPress(event);
		} else if (event.keyCode === DW_LIVEEDIT_CONSTANTS.TabKeyCode) {
			event.preventDefault();
			event.stopPropagation();
			return false;
		}
        this.selectHandler();
	}.bind(this);
    return true;
};
// This will propogate selection to textformatting and also removes the disable selection class we attached to body
TextEditHud.prototype.mouseupListenerOnWindow = function (e) {
    'use strict';
	this.isMouseDown = false;
    if (document.body.classList.contains(DW_LIVEEDIT_CONSTANTS.DisableSelectionOnBody_TextEdit)) {
        document.body.classList.remove(DW_LIVEEDIT_CONSTANTS.DisableSelectionOnBody_TextEdit);
        var classes = document.body.getAttribute("class");
        if (classes &&  classes.trim() === "") {
            document.body.removeAttribute("class");
        }
    }
    this.selectHandler();
};
// tracking the mousedown for selection on mouseout handling purposes
TextEditHud.prototype.mouseDownListenerOnElement = function (e) {
    'use strict';
    this.isMouseDown = true;
};

// when we are going out of the element disable selection for everybody else and let the selection be there for our element
TextEditHud.prototype.mouseOutListener = function (e) {
    'use strict';
    if (this.getVisibility() && this.isMouseDown) {
        if (!document.body.classList.contains(DW_LIVEEDIT_CONSTANTS.DisableSelectionOnBody_TextEdit)) {
            document.body.classList.add(DW_LIVEEDIT_CONSTANTS.DisableSelectionOnBody_TextEdit);
        }
    }
};

// createTempIdS - associates temp IDs for affected elements.
//		These temp IDs will be used to map back with new 
//		data_liveedit_tagid values from DW
TextEditHud.prototype.createTempIdS = function (node) {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : createTempIdS");
    if (!node) {
        return false;
    }

    if (node.nodeType === Node.TEXT_NODE) {
        return false;
    }

	// if node has temp ID already, we should not be adding new temp ID
	// because there may be commit process is going on and changing temp ID
	// would disturb the commit process.
	var lastTempId = this.m_dwgTempElementCount;
    if (node.nodeType === Node.ELEMENT_NODE
			&& node.tagName !== DW_LIVEEDIT_CONSTANTS.TextContainer
			&& !node.hasAttribute(DW_LIVEEDIT_CONSTANTS.DWTempId)) {
        this.m_dwgTempElementCount += 1;
        node.setAttribute(DW_LIVEEDIT_CONSTANTS.DWTempId, this.m_dwgTempElementCount);
    }

    var i, children = node.children;
    for (i = 0; i < children.length; i += 1) {
        this.createTempIdS(children[i]);
    }

	if (lastTempId !== this.m_dwgTempElementCount) {
		// tempId created
		return true;
	}
};

// updateNewIdS - updates new "data_liveedit_tagid" values 
//		using associated temp IDs.
TextEditHud.prototype.updateNewIdS = function (node, newIdInfo) {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : updateNewIdS");

    if (!node) {
        this.logDebugMsg("function begin : TextEditHud : node not present");
        return;
    }
    if (node.nodeType === Node.TEXT_NODE) {
        this.logDebugMsg("function begin : TextEditHud : node text node");

        return;
    }

    if (node.nodeType === Node.ELEMENT_NODE) {
        var dwTempId = node.getAttribute(DW_LIVEEDIT_CONSTANTS.DWTempId);
        node.removeAttribute(DW_LIVEEDIT_CONSTANTS.DWTempId);
        if (dwTempId && newIdInfo[dwTempId]) {
			this.logDebugMsg("TextEditHud : updateNewIdS - node element tempId: " + dwTempId + " new ID: " + newIdInfo[dwTempId]);
            node.setAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId, newIdInfo[dwTempId]);
        } else if (dwTempId) {
			this.logDebugMsg("Failed TextEditHud : updateNewIdS - node element tempId: " + dwTempId + " node tagName " + node.tagName);
		}
    }

    var i, children = node.children;
    for (i = 0; i < children.length; i += 1) {
        this.updateNewIdS(children[i], newIdInfo);
    }
};

// Find first text node in the inside incoming node.
TextEditHud.prototype.findAnyTextNode = function (node) {
    'use strict';

    if (!node) {
        return null;
    }

    if (node.nodeType === Node.TEXT_NODE && !this.isIgnorableText(node) && !(node.nodeValue.length === 1 && node.nodeValue.charCodeAt(0) === DW_LIVEEDIT_CONSTANTS.NBSPCharCode)) {
        return node;
    }

    if (node.nodeType === Node.ELEMENT_NODE) {
        var chidNodes = node.childNodes,
            textNode = null;

        if (chidNodes) {
            var i, tempNode = chidNodes[0];
            for (i = 0; tempNode; i += 1, tempNode = chidNodes[i]) {
                textNode = this.findAnyTextNode(tempNode);
                if (textNode) {
                    return textNode;
                }
            }
        }
    }

    return null;
};

// Find any <br> node in the inside incoming node.
TextEditHud.prototype.findAnyBRNode = function (node) {
    'use strict';

    if (!node || node.nodeType !== Node.ELEMENT_NODE) {
        return null;
    }

    if (node.tagName === "BR") {
        return node;
    }

	var chidNodes = node.childNodes,
		brNode = null;

	if (chidNodes) {
		var i, tempNode = chidNodes[0];
		for (i = 0; tempNode; i += 1, tempNode = chidNodes[i]) {
			brNode = this.findAnyBRNode(tempNode);
			if (brNode) {
				return brNode;
			}
		}
	}

    return null;
};

// Find last text node in the inside incoming node.
TextEditHud.prototype.findLastTextNode = function (node) {
    'use strict';

    if (!node) {
        return null;
    }

    if (node.nodeType === Node.TEXT_NODE) {
        return node;
    }

    if (node.nodeType === Node.ELEMENT_NODE) {
        var lastChild = node.lastChild,
            textNode = null;

        while (lastChild) {
			textNode = this.findLastTextNode(lastChild);

			if (textNode) {
				break;
			}

			lastChild = lastChild.previousSibling;
        }

		return textNode;
    }

    return null;
};

// cleanNBSPsIntroducedForEditing - We introduce &nbsp; to some element to 
// make it editable, search and remove them before quiting editable session. 
TextEditHud.prototype.cleanNBSPsIntroducedForEditing = function (element) {
	'use strict';
	if (!element || element.nodeType !== Node.ELEMENT_NODE) {
		return;
	}

	var orgHtml, dwID = element.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);
	if (!dwID) {
		if (element.tagName !== DW_LIVEEDIT_CONSTANTS.TextContainer) {
			return;
		}
	} else {
		orgHtml =  this.m_mapTempNBSPsForEditing[dwID];
	}

	if (typeof orgHtml === 'string') {
		// If we are here, then we have changed the innerHTML of this element.
		// If innerHTML is remains &nbsp; we will put back original html.
		if (element.innerHTML === DW_LIVEEDIT_CONSTANTS.NonBreakingSpace) {
			this.logDebugMsg("Reverting $nbsp; from element tag:" + element.tagName + " DWID:" + dwID);
			element.innerHTML = orgHtml;
		}
	} else if (element.childNodes) {
		var i,
			chidNodes = element.childNodes,
			node = chidNodes[0],
			curTextNode = null;

		// validate nodes
		for (i = 0; node; i += 1, node = chidNodes[i]) {
			this.cleanNBSPsIntroducedForEditing(node);
		}
	}
};

// canElementBeEditedAsText - if element is empty or has only &nbsp; then 
// we can use that as text for editing.
TextEditHud.prototype.canElementBeEditedAsText = function (element) {
	'use strict';
	if (!element) {
		return false;
	}

	var dwID = element.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);
	if (!dwID) {
		return false;
	}

	this.m_mapTempNBSPsForEditing = {};

	// if NO text node present inside element, then 
	// check whether element is empty blocks. If it is
	// empty then add &nbsp; and enable text editing.
	var canbe = false;
	if (element.nodeType === Node.ELEMENT_NODE &&
			element.innerHTML === DW_LIVEEDIT_CONSTANTS.NonBreakingSpace) {
		canbe = true;
	} else if (element.nodeType === Node.ELEMENT_NODE &&
				(DW_LIVE_TEXT_EDIT.BlockInclusionList.indexOf(element.tagName) >= 0 ||
				DW_LIVE_TEXT_EDIT.InlineTextElements.indexOf(element.tagName) >= 0) &&
				element.innerHTML.length === 0) {
		this.logDebugMsg("Adding $nbsp; to element tag:" + element.tagName + " DWID:" + dwID);
		this.m_mapTempNBSPsForEditing[dwID] = element.innerHTML;
		element.innerHTML = DW_LIVEEDIT_CONSTANTS.NonBreakingSpace;
		canbe = true;
	}

	return canbe;
};

// enterIntoTextEditing - enter into text editing for the element's text content.
TextEditHud.prototype.enterIntoTextEditing = function (element) {
	'use strict';

	this.logDebugMsg("function begin : TextEditHud : enterIntoTextEditing");

	if (!element) {
		return false;
	}

	var curTextNode = this.findAnyTextNode(element);

	// If no text node, check whether there is any <br> tag
	if (!curTextNode) {
		curTextNode = this.findAnyBRNode(element);
	}

	// Check whether element can be used as text
	var elementCopy = element.cloneNode(true);
	if ((elementCopy && elementCopy.nodeType === Node.ELEMENT_NODE) && elementCopy.tagName !== "A") {
		elementCopy = this.findAnchorNodeParentForGivenNode(elementCopy);
	}

	if (!curTextNode && this.canElementBeEditedAsText(element)) {
		curTextNode = element;
	}

	if (curTextNode) {
		this.resetContext();
		this.enableHud();
		this.m_curTextNode = curTextNode;
		this.m_curElement = null;
		this.m_elementUnderCurrentEditingSession = null;

		// If esh is on anchor tag, figure out the the reference from the curTextNode and preserve it. 
		if (this.m_nodeAtClickPoint === null) {
			var nodeIter = (curTextNode.nodeType === Node.TEXT_NODE ? curTextNode.parentNode : curTextNode);
			this.m_nodeAtClickPoint = this.findAnchorNodeParentForGivenNode(nodeIter);
			if (this.m_nodeAtClickPoint && elementCopy && elementCopy.tagName === "A") {
				this.m_nodeAtClickPointInnerHTML = elementCopy.innerHTML;
			}
		}

		this.draw({}, true);

        //execute the following code to set the selection in the Dreamweaver side only if there is a 
        //selection mismatch here and in dreamweaver
        if (window.liveViewExtensionsObject) {
            var dreamweaverSelection = window.liveViewExtensionsObject.getCurrentSelectedElement();
            if (this.m_curElement) {
                var elementUnderEdit = this.FindElementToBringbackESHHud(this.m_curElement);
                if (elementUnderEdit !== dreamweaverSelection) {
                    var dwId = elementUnderEdit.getAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId);
                    window.DWLECallJsBridgingFunction(this.getHudName(), "DWSetSelectedElement", dwId, false);
                }
            }
        }

		var textNode = this.findLastTextNode(this.m_curElement);
		if (textNode) {

			if (textNode.parentNode && this.hasSingleElementWhoseContentIsWhitespace(textNode.parentNode)) {
				this.setIbeamToGivenCharOffsetInAGivenNode(textNode, 0);
			} else {
				this.setIbeamToGivenCharOffsetInAGivenNode(textNode, textNode.nodeValue.length);
			}

		}
        if (window.liveViewExtensionsObject && window.liveViewExtensionsObject.dwObject) {
            window.liveViewExtensionsObject.dwObject.logHeadlightsData(DW_ICE_HEADLIGHTS.DWShortcuts, DW_ICE_HEADLIGHTS.StartEditing);
        }
        return true;
	}

	return false;
};

TextEditHud.prototype.getCharacterOffsetWithin = function (element) {
	'use strict';
	this.logDebugMsg("function begin : TextEditHud : getCharacterOffsetWithin");
	var caretOffset = 0,
		selection = window.getSelection();

	if (selection && element) {
		var preCaretRange = selection.getRangeAt(0),
			range = preCaretRange.cloneRange();

		range.selectNodeContents(element);
		range.setEnd(preCaretRange.endContainer, preCaretRange.endOffset);

		caretOffset = range.toString().length;
	}

    return caretOffset;
};

// recreateTextEditingContext - called after commit to recreate the dw_span tag and 
//  text group context and identifiers.
TextEditHud.prototype.recreateTextEditingContext = function () {
	'use strict';
	this.logDebugMsg("function begin : TextEditHud : recreateTextEditingContext");

	var dwSpanElement = this.m_elementUnderCurrentEditingSession;
	if (!dwSpanElement || dwSpanElement.innerHTML.length <= 0) {
		return false;
	}

	// No text node, we can't re create context
	if (!this.findAnyTextNode(dwSpanElement)) {
		return false;
	}

	// Here we will remove and recreate dw_span and current context.
	// We will find some text element which was under dw_span and 
	// use that as current text node (usually it is identified as part 
	// of setElement) and call this.draw() which will recreate dw_span 
	// and context.
	var prevSibling = dwSpanElement.previousSibling,
		nextSibling = dwSpanElement.nextSibling,
		parentNode = dwSpanElement.parentNode,
		curNode = null,
		curTextNode = null,
		curDwSpanChild =  dwSpanElement.firstChild,
		anchNode = document.getSelection().anchorNode,
		offset = this.getCharacterOffsetWithin(anchNode);

	// To recreate entire context, move out all the elemnts from 
	// dw-span tag and remove it.
	while (curDwSpanChild) {
		curDwSpanChild = dwSpanElement.removeChild(curDwSpanChild);
		parentNode.insertBefore(curDwSpanChild, dwSpanElement);
		curDwSpanChild =  dwSpanElement.firstChild;
	}
	parentNode.removeChild(dwSpanElement);

	if (prevSibling) {
		curNode = prevSibling.nextSibling;
	} else {
		curNode = parentNode.firstChild;
	}

	while (curNode) {
		if (nextSibling && curNode === nextSibling) {
			break;
		}

		curTextNode = this.findAnyTextNode(curNode);
		if (curTextNode) {
			break;
		}

		curNode = curNode.nextSibling;
	}

	if (curTextNode) {
		this.resetContext();
		this.m_curTextNode = curTextNode;
		this.m_curElement = null;
		this.m_elementUnderCurrentEditingSession = null;
		this.draw({}, true);

		// Try restore selection.
		if (!anchNode) {
			anchNode = this.findLastTextNode(this.m_curElement);
			offset = 0;
		}

		if (anchNode.parentNode && this.hasSingleElementWhoseContentIsWhitespace(anchNode.parentNode)) {
			this.setIbeamToGivenCharOffsetInAGivenNode(anchNode, 0);
		} else {
			this.setIbeamToGivenCharOffsetInAGivenNode(anchNode, offset);
		}

		return true;
	}

	return false;
};


// refreshTextEditingContext - called after commit to refresh/recalculate the
//  text group context and identifiers.
TextEditHud.prototype.refreshTextEditingContext = function () {
	'use strict';

	this.logDebugMsg("function begin : TextEditHud : refreshTextEditingContext");

	var dwSpanElement = this.m_elementUnderCurrentEditingSession;
	if (!dwSpanElement || dwSpanElement.innerHTML.length <= 0) {
		return false;
	}

	//
	// Check whether this is block level grouping.
	//
	var i,
		node,
		isBlockLevelGrouping = false,
		chidNodes = dwSpanElement.childNodes,
		curTextNode = null;

	if (chidNodes) {
		node = chidNodes[0];
		isBlockLevelGrouping = true;
		// validate nodes
		for (i = 0; node; i += 1, node = chidNodes[i]) {
			if (node.nodeType !== Node.ELEMENT_NODE || DW_LIVE_TEXT_EDIT.BlockInclusionList.indexOf(node.tagName) < 0) {
				isBlockLevelGrouping = false;
				break;
			}
		}
	}

	if (isBlockLevelGrouping) {
		// Incase of block level grouping, the current dw_span element will
		// have block level elements only. We will use any of its child text node
		// as current text and refresh the text identifies. 
		// Note: We don't need to recreate dw_span here as there is no hanging 
		// text.
		curTextNode = this.findAnyTextNode(dwSpanElement);

		// If no valid text node, consider the element as text if it can be.
		if (!curTextNode && this.canElementBeEditedAsText(dwSpanElement.firstChild)) {
			curTextNode = dwSpanElement.firstChild;
		}

		if (curTextNode) {
			this.resetContext();
			this.m_curTextNode = curTextNode;

			// The below will prepare context for just one block which contains curTextNode.
			if (!this.prepareTextIdentifier(curTextNode)) {
				return false;
			}

			// Include the all other blocks  in grouping.
			this.m_textGroupContext.group.startingSiblingText = dwSpanElement.firstChild;
			this.m_textGroupContext.group.endingSiblingText = dwSpanElement.lastChild;

			this.m_textIdentifier.groupStarting = this.getTextNodeIdentifier(this.m_textGroupContext.group.startingSiblingText);
			if (this.m_textGroupContext.group.startingSiblingText !== this.m_textGroupContext.group.endingSiblingText) {
				this.m_textIdentifier.groupEnding = this.getTextNodeIdentifier(this.m_textGroupContext.group.endingSiblingText);
			}

			// Successfully refreshed the identifiers.
			return true;
		}
	} else {
		// TODO: the same code can be used for block level grouping too. As the above code is already tested code, not changing now.
		// There is hanging text in current editing context. Try refreshing context
		curTextNode = this.findAnyTextNode(dwSpanElement);

		if (curTextNode) {
			this.resetContext();
			this.m_curTextNode = curTextNode;

			// The below will prepare context for curTextNode.
			if (!this.prepareTextIdentifier(curTextNode, true)) {
				return false;
			}

			// create Group info object
			this.m_textGroupContext.group = {};
			this.m_textGroupContext.group.originalNode = curTextNode;
			this.m_textGroupContext.group.allSiblingsAreText = dwSpanElement.parentNode.firstChild === dwSpanElement.parentNode.lastChild;

			// Include the all other blocks  in grouping.
			this.m_textGroupContext.group.startingSiblingText = dwSpanElement.firstChild;
			this.m_textGroupContext.group.endingSiblingText = dwSpanElement.lastChild;

			this.m_textIdentifier.grouped = true;
			this.m_textIdentifier.groupStarting = this.getTextNodeIdentifier(this.m_textGroupContext.group.startingSiblingText, true);
			if (this.m_textGroupContext.group.startingSiblingText !== this.m_textGroupContext.group.endingSiblingText) {
				this.m_textIdentifier.groupEnding = this.getTextNodeIdentifier(this.m_textGroupContext.group.endingSiblingText, true);
			}

			// Successfully refreshed the identifiers.
			return true;
		}
	}

	// failed to refresh the identifiers
	return false;
};

// updateDWIdForTempIDcallback - call back function for DW to refresh 
//		"data_liveedit_tagid" values of affected elements using temp IDs.
TextEditHud.prototype.updateDWIdForTempIDcallback = function (newIdObj) {
    'use strict';

    this.logDebugMsg("function begin : TextEditHud : updateDWIdForTempIDcallback");

    if (!newIdObj) {
        // This should never happen as per protocol.
        this.logDebugMsg("TextEditHud : updateDWIdForTempIDcallback - invalid argument");
        return;
    }

    var parentNode = this.m_nodeCache[newIdObj.parentId];
    if (!parentNode || newIdObj.parentSwapped) {
		if (newIdObj.parentSwapped && newIdObj.oldParentId) {
			parentNode = this.m_nodeCache[newIdObj.oldParentId];
		}

		if (!parentNode) {
			this.logDebugMsg("TextEditHud : updateDWIdForTempIDcallback - parentNode not found");
			return;
		}

		parentNode.setAttribute(DW_LIVEEDIT_CONSTANTS.DWUniqueId, newIdObj.parentId);
    }

    this.updateNewIdS(parentNode, newIdObj);
    if (window.liveViewExtensionsObject && window.liveViewExtensionsObject.dwObject) {
        window.liveViewExtensionsObject.dwObject.logHeadlightsData(DW_ICE_HEADLIGHTS.OTH_ELV, DW_ICE_HEADLIGHTS.Commit);
    }

	if (!this.m_elementUnderCurrentEditingSession) {
		return;
	}

    this.logDebugMsg("TextEditHud : updateDWIdForTempIDcallback - HTML after tempID update " + this.m_elementUnderCurrentEditingSession.outerHTML);

	// Update the original HTML
	this.m_originalInnerHTMLOfElementUnderCurrentEditingSession = this.m_elementUnderCurrentEditingSession.innerHTML;

	// After saving, we are retaining the editing session. However, commit operation would have caused
	// data-id changes in DW side and also we need to update the text editing context for the new text content.
	var refreshFailed = false;
	if (!this.refreshTextEditingContext()) {
		// If we are here, we can't refresh Text identifiers because of hanging text.
		refreshFailed = true;
	}

    if (this.saveDocumentPending) {
		this.logDebugMsg("TextEditHud : updateDWIdForTempIDcallback - save pending.");

		// We have to recreate whole dw_span editing session in case 
		// refresh context is failed.
		if (refreshFailed && !this.recreateTextEditingContext()) {
			this.destroy();
		}

        // bring up DW save dialog
        window.DWSaveDocument();
        this.saveDocumentPending = false;
    }
};

// Top level method that setup neccessary infrastructure for selecting the 
// complete text block content on triple clicks on the text content
TextEditHud.prototype.initializeTripleClickHandler = function (element) {
    'use strict';

    if (!element) {
        this.logDebugMsg("initializeTripleClickHandler - invalid element");
        return;
    }

    // We will need a click counter, the old position and two timers
    var clickCounter = 0,
        clickTimer = null,
        selectTimer = null,
        resetClick = null;

    var preservedClickPosition = {
        x: null,
        y: null
    };

    // Function that actually does the selection
    var selectAll = function () {
        //Clear the existing selection and select the complete current element.
        var selection = window.getSelection();
        if (selection && element) {
            var range = document.createRange();
            range.selectNodeContents(element);
            selection.removeAllRanges();
            selection.addRange(range);
            if (window.liveViewExtensionsObject && window.liveViewExtensionsObject.dwObject) {
                window.liveViewExtensionsObject.dwObject.logHeadlightsData(DW_ICE_HEADLIGHTS.ELV_LVE, DW_ICE_HEADLIGHTS.SelectAll);
            }
        }

        clearTimeout(selectTimer);
		// on triple click we are updating the selection so we should notify this to textFormattingHud
		this.selectHandler();
	}.bind(this);

    // Function that sets up select all.
    var initiateSelectAll = function () {
        clearTimeout(selectTimer);
        selectTimer = setTimeout(selectAll, DW_LIVE_TEXT_EDIT.Triple_Click_Selection_Threshold);
    };

    function onThirdClick(event) {
        clickCounter += 1;

        var currentClickPosition = {
            x: event.clientX,
            y: event.clientY
        };

        // identify the triple click - if three clicks happened within the Triple_Click_Position_Threshold pixels and within 
        // Triple_Click_Interval_Threshold time period
        if (clickCounter === 3 && (Math.abs(preservedClickPosition.x - currentClickPosition.x) < DW_LIVE_TEXT_EDIT.Triple_Click_Position_Threshold) && (Math.abs(preservedClickPosition.y - currentClickPosition.y) < DW_LIVE_TEXT_EDIT.Triple_Click_Position_Threshold)) {
            initiateSelectAll();
        }

        resetClick();
    }

    // Function to reset the data
    resetClick = function () {
        clickCounter = 0;
        preservedClickPosition = {
            x: null,
            y: null
        };

        element.removeEventListener("click", onThirdClick, false);
    };


    // Function to wait for the next click
    var conserveClick = function (currentClickPosition) {
        preservedClickPosition = currentClickPosition;
        clearTimeout(clickTimer);
        clickTimer = setTimeout(resetClick, DW_LIVE_TEXT_EDIT.Triple_Click_Interval_Threshold);
    };

    // simulate the triple click
    element.addEventListener('dblclick', function (event) {
        // On double click, register for thrid single click and leave it.	
        if (clickCounter === 0) {
            clickCounter = 2;
            element.addEventListener("click", onThirdClick, false);

            // Get the current mouse position
            var currentClickPosition = {
                x: event.clientX,
                y: event.clientY
            };

            conserveClick(currentClickPosition);
        }
    });

    // When click happens to be on form label, browser forcibly sets focus to controls which is sitting next to it.
    // To fix the same, we are intercepting single click within the editing block and stopping it here itself.
    element.addEventListener('click', function (event) {
        event.preventDefault();
        event.stopPropagation();
    });


};

// Unintialize triple click handler.
TextEditHud.prototype.unInitializeTripleClickHandler = function (element) {
    'use strict';

    if (!element) {
        this.logDebugMsg("unInitializeTripleClickHandler - invalid element");
        return;
    }

    // remove the click handlers from the element
    element.removeEventListener('click');
    element.removeEventListener('dblclick');
};
// updateselection from text formatting hud
TextEditHud.prototype.updateSelection = function () {
    "use strict";
    var sel = window.getSelection();
	if (sel) {
        this.m_lastSelection = sel.toString();
	}
};
// Select handler which listens to selection on the text and calls text hud
TextEditHud.prototype.selectHandler = function () {

    'use strict';
    var curSelectionString = null;
    var currentSelection = window.getSelection();
	if (currentSelection) {
		curSelectionString = currentSelection.toString();
	}
    var messageDetails = {};
    if (!curSelectionString) {
        if (this.m_lastSelection !== null) {
            messageDetails.type = DW_EXTENSION_EVENT.TEXTEDIT_SELECTION_LOST;
            dwExtensionController.sendMessage(messageDetails);
            this.m_lastSelection = null;
        }
    } else {
        messageDetails.type = DW_EXTENSION_EVENT.TEXTEDIT_SELECTION_CHANGED;
        var queryFailed = false, boldState = false, italicState = false;
        try {
            boldState = document.queryCommandState('bold');
            italicState = document.queryCommandState('italic');
        } catch (e) {
            queryFailed = true;
        }
        messageDetails.bold = boldState;
        messageDetails.italic = italicState;
        messageDetails.failed = queryFailed;
        dwExtensionController.sendMessage(messageDetails);
        this.m_lastSelection = curSelectionString;
	}
};

// Initialize cut paste handler for text edit
TextEditHud.prototype.initializeCutPasteHandler = function (element) {
    'use strict';

    if (!element) {
        this.logDebugMsg("initializeCutPasteHandler - invalid element");
        return;
    }

    element.addEventListener("paste", function (event) {
        try {
            // validate the incoming event.
            if (!event) {
                return;
            }

            //Cancel default paste operation.
            event.preventDefault();

            //Stop the event propagation.
            event.stopPropagation();

            // Get text representation of clipboard
            var text = event.clipboardData.getData("text/plain");

            // Replace new lines to br tags
            text = text.replace(/\r/g, "");
			text = text.replace(/\n/g, " ");

			// If < and > are part of the content, insertHTML tries handle them differently. It tries to add close tag
			// when <tagname> is pasted. Hence we are encoding them here.
			text = text.replace(/</g, DW_LIVEEDIT_CONSTANTS.MarkupForLessThan);
			text = text.replace(/>/g, DW_LIVEEDIT_CONSTANTS.MarkupForGreatorThan);

			// if there are multiple spaces next to each other, trim them to singe one
			text = text.replace(/\s+/g, " ");

			// Insert text manually
            document.execCommand("insertHTML", false, text);

            // Reset the feedback to consider the text change (post paste operation).
            this.showTextEditingFeedback();
        } catch (e) {
            this.logDebugMsg("TextHUD Paste handler - exception caught while pasting :" + e.message);
        }
    }.bind(this));

    element.addEventListener("cut", function (event) {
        try {
            // validate the incoming event.
            if (!event) {
                return;
            }

            //Cancel default cut operation.
            event.preventDefault();

            //Stop the event propagation.
            event.stopPropagation();

            // Cut text manually
            document.execCommand("cut");

            // Reset the feedback to consider the text change (post cut operation).
            this.showTextEditingFeedback();
        } catch (e) {
            this.logDebugMsg("TextHUD Cut handler - exception caught while pasting :" + e.message);
        }
    }.bind(this));

};

// Uninitialize cut paste handler.
TextEditHud.prototype.uninitializeCutPasteHandler = function (element) {
    'use strict';

    if (!element) {
        this.logDebugMsg("uninitializeCutPasteHandler - invalid element");
        return;
    }

    // remove the cut and paste handler from the element
    element.removeEventListener('paste');
    element.removeEventListener('cut');
};

// For showing editing disablement UI, we are firing single via Controller, that
// will in turn show the proper disablement UI.
TextEditHud.prototype.ShowEditingDisabledUI = function (element) {
    'use strict';

	if (!element) {
		return;
	}

    this.logDebugMsg("function begin : ShowEditingDisabledUI : draw" + element);

	var editDisabled = true;
	this.BringbackESHHud(element, editDisabled);
};
